import React from 'react';
import {withNaming} from '@bem-react/classname';

import {
    QuarterTableCell,
    QuarterTableCellProps,
} from '../QuarterTableCell/QuarterTableCell';

import './QuarterTableRow.css';


type QuarterTableRowType = 'header' | 'default';
type QuarterTableRowCellPurpose = 'space' | 'data';

type QuarterTableRowCellData = QuarterTableCellProps['data'];

/**
 * Более узкий интерфейс параметров компонента ячейки, потому что тип "fill" устанавливается автоматически последней
 * ячейке в строке для полного заполнения таблицы.
 */
interface TypeExcludedQuarterTableCellProps extends QuarterTableCellProps {
    type: Exclude<QuarterTableCellProps['type'], 'fill'>;
}

/**
 * Интерфейс струкруры ячейки
 */
interface QuarterTableRowCellStructure extends Omit<TypeExcludedQuarterTableCellProps, 'data'> {
    purpose?: QuarterTableRowCellPurpose;
}

/**
 * Интерфейс струкруры группы ячеек, позволяет определить группу из нескольких ячеек с выделением посредством задания
 * цвета фона
 */
export interface QuarterTableCellGroupStructure {
    cells: Array<QuarterTableRowCellStructure>;
    backgroundColor?: string;
}

type QuarterTableRowStructureItem = QuarterTableRowCellStructure | QuarterTableCellGroupStructure;
export type QuarterTableRowStructure = Array<QuarterTableRowStructureItem>;

export interface QuarterTableRowProps {
    /**
     * Содержимое ячеек строки таблицы, которое метчится на структуру
     * (заполняются только ячейки таблицы которые имеют purpose со значением "data" (по умолчанию))
     */
    data: Array<QuarterTableRowCellData>;

    /**
     * Задает структуру строки, определяет ширину ячеек строки и назначение (пропуск / отображение данных)
     */
    structure: QuarterTableRowStructure;

    /**
     * Задает цвет фона ячеек строки
     */
    backgroundCellColor?: string;

    /**
     * Задает цвет фона строки
     */
    backgroundColor?: string;
    className?: string;

    /**
     * Определяет является ли строка загаловком таблицы (type=header)
     */
    type?: QuarterTableRowType;
}

function isCellGroup(structureConfig: QuarterTableRowStructureItem): structureConfig is QuarterTableCellGroupStructure {
    return Array.isArray((structureConfig as QuarterTableCellGroupStructure).cells);
}

const cn = withNaming({e: '__', m: '_'})('quarter-table-row');

export const QuarterTableRow = React.memo<QuarterTableRowProps>(({
                                                                     data,
                                                                     structure,
                                                                     className,
                                                                     type = 'default',
                                                                     backgroundCellColor,
                                                                     backgroundColor,
                                                                 }) => {
    const getDataStructureConfig = (): Array<QuarterTableCellProps> => {
        let currentDataIndex = 0;

        const config: Array<QuarterTableCellProps> = structure.map((currentStructureElement) => {
            if (isCellGroup(currentStructureElement)) {
                return currentStructureElement.cells
                    .map((cellStructureConfig) => {
                        return ({
                            className: cn('cell', [cellStructureConfig.className]),
                            backgroundColor:
                                backgroundCellColor ||
                                cellStructureConfig.backgroundColor ||
                                currentStructureElement.backgroundColor,
                            type: cellStructureConfig.type,
                            data: cellStructureConfig.purpose === 'space' ? undefined : data[currentDataIndex++],
                        });
                    });
            }

            return ({
                className: cn('cell', [currentStructureElement.className]),
                backgroundColor: backgroundCellColor || currentStructureElement.backgroundColor,
                type: currentStructureElement.type,
                data: currentStructureElement.purpose === 'space' ? undefined : data[currentDataIndex++],
            });
        }).flat();

        return config.concat({type: 'fill', backgroundColor: backgroundCellColor, className: cn('cell')});
    };

    return (
        <div className={cn({type}, [className])}>
            {
                getDataStructureConfig().map((structureConfig, configIndex) => {
                    return (
                        <QuarterTableCell
                            key={configIndex}
                            {...structureConfig}
                        />
                    );
                })
            }
        </div>
    );
});

QuarterTableRow.displayName = 'QuarterTableRow';
