import React, {FocusEventHandler, KeyboardEventHandler, useRef} from 'react';
import {withNaming} from '@bem-react/classname';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';

type ResponsiveInputProps = React.ComponentProps<typeof Textinput> & {
    onSave?(value: string): void;
    onCancel?(value: string): void;
    cancellable?: boolean;
    saveOnBlur?: boolean;
    selectOnFocus?: boolean;
};

const cn = withNaming({e: '__', m: '_'})('responsive-input');

export const ResponsiveInput: React.VFC<ResponsiveInputProps> = ({
                                                                     className,
                                                                     onSave,
                                                                     cancellable= true,
                                                                     selectOnFocus= true,
                                                                     saveOnBlur = true,
                                                                     onChange,
                                                                     onCancel,
                                                                     value,
                                                                     onBlur,
                                                                     onKeyDown,
                                                                     ...props
                                                                 }) => {
    const initialValue = useRef<string>('' + value);

    const cancelHandler = () => {
        cancellable && onCancel && onCancel(initialValue.current);
    };

    const saveHandler = (value: string) => {
        onSave && onSave(value);
    };

    const buttonPressHandler: KeyboardEventHandler<HTMLInputElement> = (e) => {
        e.code === 'Enter' && saveHandler(e.currentTarget.value);
        e.code === 'Escape' && cancelHandler();

        if (onKeyDown) {
            onKeyDown(e);
        }
    };

    const blurHandler: FocusEventHandler<HTMLInputElement> = (e) => {
        saveOnBlur && saveHandler(e.target.value);

        if (onBlur) {
            onBlur(e);
        }
    };

    const focusHandler: FocusEventHandler<HTMLInputElement> = (e) => {
        selectOnFocus && e.target.select();
    };

    return (
        <Textinput
            className={cn('', [className])}
            onKeyDown={buttonPressHandler}
            onFocus={focusHandler}
            onBlur={blurHandler}
            value={value}
            {...props}
        />
    );
};
