import React from 'react';
import {apiAddTask, apiCall, apiEditTask, apiSearch, taskUrl} from '../../startrek';
import {Textinput} from '@yandex-lego/components/Textinput/desktop/bundle';
import {Text} from '@yandex-lego/components/Text/desktop/bundle';
import {Button} from '@yandex-lego/components/Button/desktop/bundle';
import {Checkbox} from '@yandex-lego/components/Checkbox/desktop/bundle';
import {Select} from '@yandex-lego/components/Select/desktop/bundle';
import {Link} from '@yandex-lego/components/Link/desktop/bundle';
import {Spin} from '@yandex-lego/components/Spin/desktop/bundle';
import {Tumbler as TumblerDesktop, withSizeM, withViewDefault} from '@yandex-lego/components/Tumbler/desktop';
import {compose} from '@bem-react/core';

import {intersection} from 'lodash';

import {taskCompareFn} from '../../lib/functions';
import {TaskEditor} from '../TaskEditor/TaskEditor';
import {HintLink} from '../HintLink/HintLink';
import {TaskList} from '../TaskList/TaskList';
import {RightPanel} from '../RightPanel/RightPanel';
import {TaskFrame} from '../TaskFrame/TaskFrame';
import {processTasks} from './processTasks';
import {createMd} from './taskMd';
import {Groups} from './Groups';
import {Stories} from './Stories';
import {loadParentsRec} from '../../lib/loadParentsRec';

import './SprintPlanner.css';
import {Highlight} from "../Highlight/Highlight";


const Tumbler = compose(withSizeM, withViewDefault)(TumblerDesktop);

const hasAllTags = (what, where) => {
    const whatCi = what.map((tag) => tag.toLowerCase());
    const whereCi = (where || []).map((tag) => tag.toLowerCase());
    return intersection(whatCi, whereCi).length === what.length;
};

const getTaskTeam = (task, sprint, teams) => {
    if (task.team) {
        return task.team;
    }
    if (sprint) {
        let sprintOk = false;
        for (let taskSprintObj of task.sprint || []) {
            if (taskSprintObj.display === sprint) {
                sprintOk = true;
                break;
            }
        }
        if (!sprintOk) {
            return null;
        }
    }
    for (let team of teams) {
        if (hasAllTags([...team.tags, ...team.assignedTags], task.tags)) {
            return team.name;
        }
    }
    return null;
};

const getTaskPreTeam = (task, teams) => {
    for (let team of teams) {
        if (hasAllTags(team.tags, task.tags)) {
            return team.name;
        }
    }
    return null;
};

export class SprintPlanner extends React.Component {
    constructor(props) {
        super(props);
        this.state = {
            tasks: null,
            filteredTasks: [],
            locked: false,
            data: processTasks([]),
            statusMessage: null,
            teamFilter: null,
            totals: [],
            newTaskMode: false,
            newTaskName: '',
            //sprint: '2021q1s5',
            withoutStories: this.props.withoutStories,
            onlyAssigned: false,
            onlyUnassigned: false,
            onlyWithoutSP: false,
            //assignTags: 'ui_prod_backlog, UI, infra',
            locks: {},
            newTask: null,
            rewriteTask: {},
            autoUpdate: this.props.autoUpdate,
            freezeKeys: [],
            groupBy: 'stories',
            hideAssigneeTotals: this.props.hideAssigneeTotals,
        };
        this.sprintCache = {};
        this.rewriteTimers = {};
    }

    onQueryChange = (event) => {
        this.setState({query: event.target.value});
    };

    showStatus = (text) => {
        this.setState({statusMessage: text});
        if (this.statusTimer) {
            window.clearTimeout(this.statusTimer);
            this.statusTimer = null;
        }
        this.statusTimer = window.setTimeout(() => {
            this.setState({statusMessage: null});
        }, 1000);
    };

    setTasks = (tasks, newState = {}, parents = null) => {
        if (parents === null) {
            parents = this.state.parents || {};
        }
        const state = {...this.state, ...newState};
        for (let i of Object.getOwnPropertyNames(tasks)) {
            let key = tasks[i].key;
            if (state.rewriteTask[key] && state.rewriteTask[key].version > tasks[i].version) {
                tasks[i] = state.rewriteTask[key];
            }
        }
        const freezeKeys = state.freezeKeys;

        const sprint = this.props.assignSprint;
        let filteredTasks = tasks;
        for (let task of tasks) {
            if (!('team' in task)) {
                task.team = getTaskTeam(task, sprint, this.props.teams);
            }
            task.preTeam = getTaskPreTeam(task, this.props.teams);
        }

        if (state.teamFilter) {
            const teamFilter = state.teamFilter;
            filteredTasks = filteredTasks.filter(
                (task) => freezeKeys.includes(task.key) || task.team === teamFilter || task.preTeam === teamFilter,
            );
        }
        if (state.onlyAssigned) {
            filteredTasks = filteredTasks.filter((task) => freezeKeys.includes(task.key) || !!task.team);
        }
        if (state.onlyUnassigned) {
            filteredTasks = filteredTasks.filter((task) => freezeKeys.includes(task.key) || !task.team);
        }
        if (state.onlyWithoutTeam) {
            filteredTasks = filteredTasks.filter((task) => freezeKeys.includes(task.key) || !task.preTeam);
        }
        if (state.onlyWithoutSP) {
            filteredTasks = filteredTasks.filter((task) => freezeKeys.includes(task.key) || !('storyPoints' in task));
        }
        if (state.useFilterAssignee) {
            if (!state.filterAssignee) {
                filteredTasks = filteredTasks.filter((task) => freezeKeys.includes(task.key) || !('assignee' in task));
            } else {
                const assigneeId = state.filterAssignee.id;
                filteredTasks = filteredTasks.filter(
                    (task) => freezeKeys.includes(task.key) || (task.assignee && task.assignee.id === assigneeId),
                );
            }
        }

        const filteredData = processTasks(filteredTasks, state.groupBy, parents);
        const data = processTasks(tasks, state.groupBy, parents);
        const mdText = createMd(data);
        filteredData.assigneeTotals = data.assigneeTotals;

        const newFreezeKeys = freezeKeys.length === 0 ? filteredTasks.map((t) => t.key) : freezeKeys;
        console.log(filteredData);

        filteredTasks.sort(taskCompareFn);

        if (this.state.tasks) {
            const indexOldTasks = {};
            const taskHash = (task) => {
                if (task.key) {
                    return [
                        task.key,
                        task.status.display,
                        task.assignee && task.assignee.display,
                        task.storyPoints,
                        task.summary,
                        task.weight,
                    ].join(':');
                } else {
                    return;
                }
            };
            for (const task of this.state.tasks) {
                indexOldTasks[taskHash(task)] = 1;
            }
            for (const task of tasks) {
                if (!indexOldTasks[taskHash(task)]) {
                    task.highlight = true;
                }
            }
        }

        if (this.highlightTimer) {
            clearTimeout(this.highlightTimer);
            this.highlightTimer = null;
            console.log('clear highlight timer');
        }
        if (!tasks.every((task) => !task.highlight)) {
            console.log('set clear highlight timer');
            this.highlightTimer = setTimeout(() => {
                const tasks = this.state.tasks;
                console.log('clear highlight. tasks: ', tasks);
                tasks.forEach((task) => delete task.highlight);
                setTimeout(() => {
                    this.setTasks([...tasks], {});
                }, 1);
            }, 1000);
        }

        const allocationParents = filteredData.stories.map(({parent: {key, display}}) => ({key, label: display}));
        console.log('в какие истории можно распределить', allocationParents);

        this.setState({
            ...newState,
            data: filteredData,
            freezeKeys: newFreezeKeys,
            tasks,
            allocationParents,
            parents,
            filteredTasks,
            mdText,
            totals: data.totals,
        });
    };

    componentDidUpdate(prevProps, prevState, snapshot) {
        let needUpdate = false;
        if (this.props.query !== prevProps.query) {
            needUpdate = true;
        }
        if (this.props.assignSprint !== prevProps.assignSprint) {
            needUpdate = true;
        }
        if (needUpdate) {
            this.onUpdate();
        }
    }

    onExport = async () => {
        this.showStatus('Выгрузка в ' + this.props.exportTicket);
        await apiEditTask(this.props.exportTicket, {description: this.state.mdText});
        this.showStatus('Выгружено');
        //console.log(this.state.mdText);
    };

    onUpdate = async () => {
        const {query = ''} = this.props;
        if (!query) {
            return;
        }
        this.setState({locked: true});
        const newQuery = query.indexOf('"$sprint"') > -1 ? query.replace(/\$sprint/g, this.props.assignSprint) : query;
        const tasks = await apiSearch(newQuery);
        this.setState({freezeKeys: []});

        const keysObj = {};
        for (let task of tasks) {
            keysObj[task.key] = true;
        }
        for (let i in this.state.rewriteTask) {
            let task = this.state.rewriteTask[i];
            if (!(task.key in keysObj)) {
                tasks.push(task);
            }
        }

        const parentKeys = new Set();
        for (const task of tasks) {
            if (task.parent) {
                parentKeys.add(task.parent.key);
            }
        }
        let parents = {};
        if (parentKeys.size) {
            await loadParentsRec(parentKeys, parents);
        }
        console.log('parents', parents);

        this.setTasks(tasks, {}, parents);
        this.setState({locked: false});

        if (this.props.exportTicket && this.state.autoExport) {
            window.setTimeout(() => {
                this.onExport();
            }, 100);
        }
    };

    componentDidMount() {
        this.onUpdate();
        // this.splitFormatter = new window.Ya.FormatterViewPort({
        //      domElem: this.refs.preview
        // });
        this.timer = window.setInterval(() => {
            if (this.state.autoUpdate) {
                this.onUpdate();
            }
        }, 5000);
    }

    setTeamFilter(team) {
        this.setState({
            teamFilter: team,
            useFilterAssignee: false,
            freezeKeys: [],
        });
        setTimeout(() => this.setTasks(this.state.tasks));
    }

    onChangeGroupBy = (e) => {
        console.log('onChangeGroupBy');
        this.setTasks(this.state.tasks, {
            groupBy: e.target.value,
        });
    };

    onShowOnlyAssigned = (mode) => {
        this.setTasks(this.state.tasks, {
            onlyAssigned: mode,
            onlyUnassigned: false,
            onlyWithoutTeam: false,
            onlyWithoutSP: false,
            freezeKeys: [],
        });
    };

    onShowOnlyUnassigned = (mode) => {
        this.setTasks(this.state.tasks, {
            onlyUnassigned: mode,
            onlyAssigned: false,
            onlyWithoutTeam: false,
            onlyWithoutSP: false,
            freezeKeys: [],
        });
    };

    onShowOnlyWithoutTeam = (mode) => {
        this.setTasks(this.state.tasks, {
            onlyUnassigned: false,
            onlyAssigned: false,
            onlyWithoutTeam: mode,
            onlyWithoutSP: false,
            freezeKeys: [],
        });
    };

    onShowOnlyWithoutSP = (mode) => {
        this.setTasks(this.state.tasks, {
            onlyUnassigned: false,
            onlyAssigned: false,
            onlyWithoutTeam: false,
            onlyWithoutSP: mode,
            freezeKeys: [],
        });
    };

    setFilterAssignee = (use, assignee) => {
        const newState = {
            onlyUnassigned: false,
            onlyAssigned: false,
            onlyWithoutTeam: false,
            onlyWithoutSP: false,
            teamFilter: null,
            useFilterAssignee: use,
            filterAssignee: assignee,
            freezeKeys: [],
        };
        this.setTasks(this.state.tasks, newState);
    };

    onAssignTask = async (task, teamName) => {
        console.log('onAssignTask: ', task, teamName);
        let team = null;
        for (let t of this.props.teams) {
            if (t.name === teamName) {
                team = t;
                break;
            }
        }
        console.log('team: ', team);

        let newTags;
        if (team && teamName) {
            newTags = [...new Set([...(task.tags || []), ...team.tags, ...team.assignedTags])];
        } else if (!teamName && task.team) {
            const taskTeam = this.props.teams.filter((team) => team.name === task.team)[0];
            newTags = new Set(task.tags);
            for (const tag of taskTeam.assignedTags) {
                if (!(tag in taskTeam.tags)) {
                    newTags.delete(tag);
                }
            }
            newTags = [...newTags];
            console.log('delete tags, new tags: ', newTags);
        }
        //this.updateSingleTask(task);

        const newProps = {
            tags: newTags,
        };

        if (this.props.assignSprint) {
            const taskSprint = task.sprint || [];
            const sprintObj = await this.getSprint();
            const currentSprintId = '' + sprintObj.id;

            if (team) {
                const newSprints = [];
                console.log('sprintObj', sprintObj);
                for (const s of taskSprint) {
                    const taskSprintObj = await apiCall(`sprints/${s.id}`);
                    console.log('taskSprintObj', taskSprintObj);
                    if (+taskSprintObj.board.id !== sprintObj.boardId) {
                        newSprints.push(s);
                    }
                }
                newSprints.push(sprintObj);
                newProps.sprint = newSprints;
                if (newProps.tags && newProps.tags.indexOf(this.props.sprintCandidateTag) > -1)
                    newProps.tags.splice(newProps.tags.indexOf(this.props.sprintCandidateTag), 1);
            } else {
                if (this.props.unsetSprint) newProps.sprint = taskSprint.filter(({id}) => id !== currentSprintId);
                if (this.props.setSprintCandidateTag) {
                    if (newProps.tags && !newProps.tags.includes(this.props.sprintCandidateTag))
                        newProps.tags.push(this.props.sprintCandidateTag);
                }
            }
        }
        console.log('task new props', newProps);

        for (const [k, v] of Object.entries(newProps)) {
            task[k] = v;
        }
        task.team = team;

        try {
            await this.onEditSave(task, newProps);
        } catch (e) {
            console.log(e);
        }

        //this.setTasks([...this.state.tasks]);

        // const updatedTask = await apiEditTask(task.key, {
        //     tags: newTags
        // });
        // this.updateSingleTask(updatedTask);
        // console.log(task.team);
    };

    getSprint = async (sprint) => {
        if (!sprint) {
            sprint = this.props.assignSprint;
        }
        if (this.sprintCache[sprint]) {
            return this.sprintCache[sprint];
        }
        let sprintList = await apiCall('sprints/_suggest?input=' + sprint);
        console.log(sprintList);
        sprintList = sprintList.filter((s) => s.name === sprint);
        if (sprintList.length !== 1) {
            this.showStatus('Неверный спринт');
            console.log('sprintList (after filtering board 1338): ', sprintList);
            return;
        }
        const sprintObj = sprintList[0];
        this.sprintCache[sprint] = sprintObj;
        return sprintObj;
    };

    onSaveNewTask = async (task, taskProps, allocationParent) => {
        console.log('onSaveNewTask task: ', task, 'taskProps:', taskProps, 'allocationParent:', allocationParent);

        const sprintObj = await this.getSprint();
        //taskProps.queue = 'ADFOX';
        if (sprintObj) {
            taskProps.sprint = [sprintObj];
        }
        const match = this.props.query.match(/"Parent Issue":\s*([A-Z]+-[0-9]+)/i);
        if (match) {
            taskProps.parent = {key: match[1]};
        }
        const newTask = await apiAddTask(taskProps);
        newTask.highlight = true;

        if (!newTask) {
            throw new Error('Error creating new task');
        }
        this.setState({
            newTaskMode: false,
        });
        this.setTasks([...this.state.tasks, newTask]);
        console.log(newTask);
        this.addTaskRewrite(newTask);
    };

    updateSingleTask = (task) => {
        // const {newTask = {}} = this.state;

        const newTasks = this.state.tasks.map((storedTask) => (storedTask.key === task.key ? task : storedTask));
        let newParents = this.state.parents;
        if (this.state.parents[task.key]) {
            newParents = {
                ...this.state.parents,
                [task.key]: task,
            };
        }

        this.setTasks(newTasks, {}, newParents);
        // if (newTask && newTask.key === task.key) {
        //     this.setState({newTask: task});
        // }
    };

    addTaskRewrite = (task) => {
        this.setState({
            rewriteTask: {
                ...this.state.rewriteTask,
                [task.key]: task,
            },
        });
        if (this.rewriteTimers[task.key]) {
            clearTimeout(this.rewriteTimers[task.key]);
            delete this.rewriteTimers[task.key];
        }
        this.rewriteTimers[task.key] = setTimeout(() => {
            const rewriteTask = this.state.rewriteTask;
            delete rewriteTask[task.key];
            delete this.rewriteTimers[task.key];
            this.setState({
                rewriteTask: {
                    ...rewriteTask,
                },
            });
        }, 15000);
    };

    onEditSave = async (task, newFields) => {
        console.log('onEditSave task: ', task, ' newFields:', newFields);
        let newTask;
        try {
            newTask = await apiEditTask(task.key, newFields);
            newTask.highlight = true;
        } catch (e) {
            console.log(e);
            return;
        }
        this.addTaskRewrite(newTask);
        this.updateSingleTask(newTask);
    };

    onNewTaskBegin = () => {
        if (this.state.newTaskMode) {
            this.setState({newTaskMode: false});
            return;
        }
        const newTask = {
            tags: this.props.assignTags,
        };
        this.setState({
            newTaskMode: true,
            newTask,
        });
    };

    onAutoUpdate = () => {
        const mode = !this.state.autoUpdate;
        this.setState({
            autoUpdate: mode,
            autoExport: false,
        });
        if (this.disableAutoUpdateTimer) {
            window.clearTimeout(this.disableAutoUpdateTimer);
            this.disableAutoUpdateTimer = null;
        }
        if (mode) {
            this.disableAutoUpdateTimer = window.setTimeout(() => {
                this.setState({
                    autoUpdate: false,
                    autoExport: false,
                });
            }, 3600000);
        }
    };

    resetFilters = () => {
        const newState = {
            onlyUnassigned: false,
            onlyAssigned: false,
            onlyWithoutTeam: false,
            teamFilter: null,
            useFilterAssignee: false,
            filterAssignee: null,
            freezeKeys: [],
        };
        this.setState(newState);
        this.setTasks(this.state.tasks, newState);
    };

    getTeamTotalsData() {
        const totals = this.state.totals;
        const teams = this.props.teams;
        const teamPlannedSp = {};
        for (const {team, storyPoints} of totals) {
            teamPlannedSp[team] = storyPoints;
        }
        const result = [];
        for (const t of teams) {
            result.push({
                team: t.name,
                planned: teamPlannedSp[t.name] || 0,
                capacity: t.capacity,
            });
        }
        return result;
    }

    onOpenTask = (task) => {
        this.setState({
            taskPanelVisible: true,
            taskPanelTask: task,
        });
    };

    hideTask = () => {
        this.setState({
            taskPanelVisible: false,
        });
    };

    setShowAssignees = (hideAssigneeTotals) => {
        this.setState({hideAssigneeTotals});
    };

    render() {
        const {data, statusMessage, teamFilter} = this.state;
        const {groups, stories} = data;
        return (
            <div className="sprint-planner">
                {statusMessage && <div className="status-message">{statusMessage}</div>}

                {this.state.taskPanelVisible && (
                    <RightPanel onClose={this.hideTask}>
                        <TaskFrame task={this.state.taskPanelTask} taskKey={this.state.taskPanelTask.key} />
                    </RightPanel>
                )}

                <div className="main-area">
                    <div className="main-area-right">
                        <div className="main-area-right-content">
                            <div className="totals">
                                {this.getTeamTotalsData().map(({team, planned, capacity}) => (
                                    <div className="totals-team" key={team}>
                                        {team}:&nbsp;
                                        <HintLink
                                            className={planned > capacity ? 'overdrive' : ''}
                                            onClick={() => this.setTeamFilter(team)}
                                        >
                                            <Highlight>{planned}</Highlight> / <Highlight>{capacity}</Highlight> sp
                                        </HintLink>
                                    </div>
                                ))}
                            </div>
                            <br />
                            <div>
                                <HintLink onClick={this.resetFilters}>
                                    <b>Все задачи</b>
                                </HintLink>
                            </div>
                            <br />
                            {!this.props.hideAssigneeTotals && (
                                <div className="customTumbler">
                                    <Tumbler
                                        view="default"
                                        size="m"
                                        checked={!this.state.hideAssigneeTotals}
                                        labelBefore="по&nbsp;людям"
                                        id="showPeopleStat"
                                        onChange={() => this.setShowAssignees(!this.state.hideAssigneeTotals)}
                                    />
                                </div>
                            )}
                        </div>
                        {(!this.state.hideAssigneeTotals || this.props.exportTicket) && (
                            <div className="main-area-right-content2">
                                {!this.props.hideAssigneeTotals &&
                                    this.state.data.assigneeTotals.map(({assignee, total}) => (
                                        <div key={assignee ? assignee.login : 'no'}>
                                            <span className="login">
                                                {assignee ? assignee.display : 'Без исполнителя'}
                                            </span>
                                            &nbsp;-&nbsp;
                                            <HintLink onClick={() => this.setFilterAssignee(true, assignee)}>
                                                <Highlight>{total}</Highlight> sp
                                            </HintLink>
                                        </div>
                                    ))}
                                <br />
                                {this.props.exportTicket && (
                                    <>
                                        Выгрузить в{' '}
                                        <Link href={taskUrl(this.props.exportTicket)}>{this.props.exportTicket}</Link>
                                        <br />
                                        <br />
                                        <Button view="default" size="s" onClick={this.onExport}>
                                            Выгрузить
                                        </Button>
                                        <br />
                                        <br />
                                        {this.state.autoUpdate && (
                                            <Checkbox
                                                label="Авто выгрузка"
                                                size="m"
                                                view="default"
                                                checked={this.state.autoExport}
                                                onChange={() => this.setState({autoExport: !this.state.autoExport})}
                                            />
                                        )}
                                    </>
                                )}
                            </div>
                        )}
                    </div>
                    <div className="main-area-left">
                        <div className="main-area-left-name">
                            {this.props.name}
                            &nbsp;&nbsp;
                            <Link size="xs" theme="ghost" href={'/'}>
                                |&nbsp;&nbsp;Список досок
                            </Link>
                        </div>
                        {/*{this.state.locked &&*/}
                        {/*<div className='sprint-planner-lock'>*/}
                        {/*    <Spin progress view="default" size="m" />*/}
                        {/*</div>*/}
                        {/*}*/}
                        <div id="startrek-query">
                            <Text>Запрос для Startrek:</Text>
                            <Textinput
                                size="s"
                                view="default"
                                value={this.props.inputQuery}
                                alt={this.props.inputQuery}
                                onKeyPress={(e) => (e && e.code && e.code === 'Enter' ? this.props.onApply() : null)}
                                onBlur={this.props.onApply}
                                onChange={(e) => this.props.configChange(e)}
                            />

                            <Button
                                view="action"
                                size="s"
                                onClick={() => {
                                    this.props.onApply();
                                }}
                            >
                                Применить
                            </Button>
                        </div>
                        <br />
                        <Button view="default" size="s" onClick={this.onNewTaskBegin}>
                            Добавить задачу
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Button view="action" size="s" onClick={this.onUpdate} disabled={this.state.locked}>
                            Обновить
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Checkbox
                            label="Авто обновление (на 1 час)"
                            size="m"
                            view="default"
                            onChange={this.onAutoUpdate}
                            checked={this.state.autoUpdate}
                        />
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Select
                            size="s"
                            theme="normal"
                            onChange={this.onChangeGroupBy}
                            value={this.state.groupBy}
                            options={[
                                {value: 'stories', content: 'По историям'},
                                {value: 'components', content: 'По компонентам'},
                            ]}
                        />
                        <br />
                        <br />
                        {this.state.newTaskMode && (
                            <TaskEditor
                                task={this.state.newTask}
                                allocationParents={this.state.allocationParents}
                                queues={this.props.newTaskQueues}
                                onSave={this.onSaveNewTask}
                                onCancel={() => this.setState({newTaskMode: false})}
                            />
                        )}
                        {/*{!this.state.newTaskMode && this.state.newTask && this.state.newTask.key &&*/}
                        {/*<TaskLine task={this.state.newTask} onAssignTask={this.onAssignTask} onEditSave={this.onEditSave} />*/}
                        {/*}*/}
                        <div className="filters">
                            Опции:&nbsp;
                            {teamFilter && (
                                <div className="filter">
                                    Только {teamFilter}&nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.setTeamFilter()}>
                                        убрать
                                    </Button>
                                </div>
                            )}
                            {this.state.withoutStories && (
                                <div className="filter">
                                    Без историй&nbsp;
                                    <Button
                                        view="clear"
                                        size="xs"
                                        onClick={() => this.setState({withoutStories: false})}
                                    >
                                        отменить
                                    </Button>
                                </div>
                            )}
                            {this.state.onlyAssigned && (
                                <div className="filter">
                                    Только взятые&nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.onShowOnlyAssigned(false)}>
                                        отменить
                                    </Button>
                                </div>
                            )}
                            {this.state.onlyUnassigned && (
                                <div className="filter">
                                    Только не взятые&nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.onShowOnlyUnassigned(false)}>
                                        отменить
                                    </Button>
                                </div>
                            )}
                            {this.state.onlyWithoutTeam && (
                                <div className="filter">
                                    Только не размеченные&nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.onShowOnlyWithoutTeam(false)}>
                                        отменить
                                    </Button>
                                </div>
                            )}
                            {this.state.onlyWithoutSP && (
                                <div className="filter">
                                    Только без SP&nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.onShowOnlyWithoutSP(false)}>
                                        отменить
                                    </Button>
                                </div>
                            )}
                            {this.state.useFilterAssignee && (
                                <div className="filter">
                                    {!this.state.filterAssignee && 'Без исполнителя'}
                                    {this.state.filterAssignee && (
                                        <span>Исполнитель: {this.state.filterAssignee.display}</span>
                                    )}
                                    &nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.setFilterAssignee(false, null)}>
                                        отменить
                                    </Button>
                                </div>
                            )}
                        </div>
                        <br />
                        <Button view="clear" size="xs" onClick={() => this.setState({withoutStories: true})}>
                            Показать без историй
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Button view="clear" size="xs" onClick={() => this.onShowOnlyAssigned(true)}>
                            Показать только взятые
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Button view="clear" size="xs" onClick={() => this.onShowOnlyUnassigned(true)}>
                            Показать только не взятые
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Button view="clear" size="xs" onClick={() => this.onShowOnlyWithoutTeam(true)}>
                            Показать не размеченные
                        </Button>
                        &nbsp;&nbsp;&nbsp;&nbsp;
                        <Button view="clear" size="xs" onClick={() => this.onShowOnlyWithoutSP(true)}>
                            Показать без SP
                        </Button>
                        <br />
                        <br />
                        <div className="sprint-planner-content">
                            {this.state.withoutStories && (
                                <TaskList
                                    tasks={this.state.filteredTasks}
                                    taskLineProps={{
                                        showWeight: true,
                                        showTags: this.props.showTags,
                                        onEditSave: this.onEditSave,
                                        onAssignTask: this.onAssignTask,
                                    }}
                                />
                            )}
                            {!this.state.withoutStories && (
                                <>
                                    <Groups
                                        showWeight={this.props.showWeight}
                                        showTags={this.props.showTags}
                                        groups={groups}
                                        allocationParents={this.state.allocationParents}
                                        onAssignTask={this.onAssignTask}
                                        onEditSave={this.onEditSave}
                                        onOpenTask={this.onOpenTask}
                                    />
                                    <Stories
                                        showWeight={this.props.showWeight}
                                        showTags={this.props.showTags}
                                        stories={stories}
                                        onAssignTask={this.onAssignTask}
                                        onEditSave={this.onEditSave}
                                        onOpenTask={this.onOpenTask}
                                    />
                                </>
                            )}
                        </div>
                        {this.state.locked && (
                            <div id="FetchDataSpinner">
                                <Spin progress view="default" size="m" />
                            </div>
                        )}
                    </div>
                </div>
                <br />
                <br />
                <br />
                <br />
            </div>
        );
    }
}
