import React from 'react';
import {apiCall} from "../../startrek";
import Autosuggest from "react-autosuggest";
import {Button} from "@yandex-lego/components/Button/desktop/bundle";

import './Suggest.css';

/**
 * params:
 *   resource - startrek resource: users, sprints etc
 *   valueProp - ключ из объекта ответа suggest-а, который будет показан как введенный текст
 *   displayClass - класс эл-та в саггестах и в режиме отображения
 *   value - значение в формате саггеста
 *   onChange - принимает объект саггеста
 *
 **/
export class Suggest extends React.PureComponent {
    constructor(props) {
        super(props);
        let values = props.value;
        if (!(props.value instanceof Array)) {
            values = props.value ? [props.value] : [];
        }
        let displayValues = values.map(v => {
            return typeof v === 'object' ? this.getSuggestionValue(v) : '' + v
        });
        const inputValue = props.multiple ? '' : displayValues[0] || '';
        this.state = {
            values,
            showInput: false,
            inputValue,
            displayValues,
            suggestions: [],
            selectedSuggestion: null,
        }
    }

    onInputChange = (e, {newValue}) => {
        this.setState({inputValue: newValue});
    }

    onSuggestionsFetchRequested = async ({ value }) => {
        const {resource, queryParams} = this.props;
        const encodedInput = encodeURIComponent(value);

        let path = `${resource}/_suggest?input=${encodedInput}`;

        if (queryParams) {
            path += Object.entries(queryParams).reduce((queryPath, [key, value]) => `${queryPath}&${key}=${value}`, '');
        }

        const suggestResult = await apiCall(path);
        this.setState({suggestions: suggestResult});
    };

    onSuggestionsClearRequested = () => {
        this.setState({
            suggestions: [],
        })
    }

    getSuggestionValue = (suggestion) => {
        if (!suggestion) {
            return '';
        }
        if (typeof suggestion === 'string') {
            return suggestion;
        }
        if (this.props.valueProp) {
            if (this.props.additionalProps) {
                const additionalInfo = ' (' + this.props.additionalProps.map(prop => suggestion[prop]).join() + ')';
                return '' + suggestion[this.props.valueProp] + additionalInfo;
            }
            return '' + suggestion[this.props.valueProp];
        }
        if ('name' in suggestion) {
            return suggestion.name;
        }
        return suggestion.display;
    }

    renderSuggestion = (suggestion) => {
        return (
            <span className={this.props.displayClass || ''}>
                {this.getSuggestionValue(suggestion)}
            </span>
        );
    }

    onSuggestionSelected = (e, {suggestion}) => {
        if (this.props.multiple) {
            this.setState({
                displayValues: [
                    ...this.state.displayValues,
                    this.getSuggestionValue(suggestion)
                ],
                values: [
                    ...this.state.values,
                    suggestion
                ],
                inputValue: ''
            });
        } else {
            this.setState({
                selectedSuggestion: suggestion,
                values: [suggestion],
                showInput: false,
                isEmpty: false,
            })
            this.props.onChange && this.props.onChange(suggestion);
        }
    }

    onDelete = () => {
        this.setState({
            inputValue: '',
            showInput: false,
            selectedSuggestion: null,
            values: [],
        })
        this.props.onChange && this.props.onChange(this.props.multiple ? [] : null);
    }

    removeMultiValue = (key) => {
        this.setState({
            values: this.state.values.filter((x, i) => i !== key)
        });
    }

    onSave = () => {
        this.setState({
            showInput: false
        })
        this.props.onChange(this.state.values);
    }

    onEdit = () => {
        this.setState({
            showInput: true,
            rollbackInputValue: this.state.inputValue,
        });
    }

    onCancel = () => {
        this.setState({
            showInput: false,
            inputValue: this.state.rollbackInputValue,
        });
    }

    render() {
        const {inputValue, suggestions} = this.state;
        if (this.props.label === 'Планировать спринт') {
            console.log('inputValue: [' + inputValue + ']');
        }
        return (
            <span className='field'>
                {this.state.showInput &&
                <span className='field-edit-content'>
                    {this.props.label && this.props.label + ': '}
                    <span>
                        <Autosuggest
                            suggestions={suggestions}
                            onSuggestionsFetchRequested={this.onSuggestionsFetchRequested}
                            getSuggestionValue={this.getSuggestionValue}
                            renderSuggestion={this.renderSuggestion}
                            onSuggestionSelected={this.onSuggestionSelected}
                            onSuggestionsClearRequested={this.onSuggestionsClearRequested}
                            inputProps={{
                                placeholder: this.props.placeholder || '',
                                value: inputValue,
                                onChange: this.onInputChange,
                                style: {width: (this.props.width || 200) + 'px'},
                                autoFocus: true,
                            }}
                        />
                        &nbsp;&nbsp;
                        {this.props.multiple &&
                        <>
                            <Button view="action" size="s" onClick={this.onSave}>
                                Сохранить
                            </Button>
                            &nbsp;&nbsp;
                        </>
                        }
                        <Button view="clear" size="xs" onClick={this.onCancel}>
                            отмена
                        </Button>
                        {this.props.multiple && this.state.values.length > 0 &&
                        <>
                            <br/><br/>
                            {this.state.values.map((v, i) =>
                                <span className='suggest-multivalue-item' key={i}>
                                    {v}
                                    &nbsp;
                                    <Button view="clear" size="xs" onClick={() => this.removeMultiValue(i)}>
                                        ✕
                                    </Button>
                                </span>
                            )}
                        </>
                        }
                    </span>
                </span>
                }
                {!this.state.showInput &&
                <span>
                    {this.props.label && this.props.label + ': '}
                    {this.state.values.map((v, i) =>
                    <span className='suggest-multivalue-item' key={i}>
                        {this.renderSuggestion(v)}
                        &nbsp;&nbsp;
                    </span>
                    )}
                    &nbsp;&nbsp;
                    <Button view="clear" size="xs" onClick={this.onEdit}>
                        изменить
                    </Button>
                    &nbsp;&nbsp;
                    {this.state.values.length > 0 &&
                    <Button view="clear" size="xs" onClick={this.onDelete}>
                        удалить
                    </Button>
                    }
                </span>
                }
            </span>
        )
    }
}
