import React from 'react';
import PropTypes from 'prop-types';
import {Button} from "@yandex-lego/components/Button/desktop/bundle";
import {Link} from "@yandex-lego/components/Link/desktop/bundle";
import {taskUrl} from "../../startrek";
import {TaskEditor} from "../TaskEditor/TaskEditor";
import {Highlight} from "../Highlight/Highlight";

const arrayIntersect = (a, b) => {
    return b.filter(x => a.includes(x));
}

export class TaskLine extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = {
            editMode: false
        }
    }

    onAssignTask = () => {
        this.props.onAssignTask && this.props.onAssignTask(this.props.task, this.props.task.preTeam);
    }

    onUnassignTask = () => {
        this.props.onAssignTask && this.props.onAssignTask(this.props.task, null);
    }

    onDump = () => {
        console.log(this.props.task);
    }

    onOpenTask = (e) => {
        if (this.props.onOpenTask) {
            e.preventDefault();
            this.props.onOpenTask(this.props.task);
        }
    }

    onEditBegin = () => {
        this.setState({
            editMode: true,
        })
    }

    onEditSave = (task, newProps) => {
        this.props.onEditSave(task, newProps);
        this.setState({
            editMode: false
        })
    }

    render() {
        const {task} = this.props;
        const {key, status, assignee = {id: ''}, summary} = task;
        let storyPoints = task.storyPoints || '?';
        if (this.props.useOriginalStoryPoints && task.originalStoryPoints) {
            storyPoints = task.originalStoryPoints;
        }
        const {team, preTeam} = task;
        let c = 'task-line';
        if (this.props.highlight) {
            c += ' task-line-highlight';
        }
        return (
            <div className={c} key={key}>
                {this.props.mode !== 'analyze' &&
                <>
                    <span className='task-team'>
                        {team}
                    </span>
                    <span className='task-team-options'>
                        {!team && preTeam &&
                        <Button view="clear" size="xs" onClick={this.onAssignTask}>
                            Взять в {preTeam}
                        </Button>
                        }
                        {team &&
                        <Button view="clear" size="xs" onClick={this.onUnassignTask}>
                            Не брать
                        </Button>
                        }
                    </span>
                    &nbsp;
                </>
                }
                <span className='task-sp'>
                    <Highlight>{storyPoints}</Highlight> sp
                </span>
                {this.props.showType &&
                <>
                    &nbsp;
                    <span className={'task-type task-type-' + task.type.key}>
                        {task.type.display}
                    </span>
                </>
                }
                &nbsp;
                <span className='task-line-task'>
                    <Link className='task-link' target="_blank" size="xs" view="default" href={taskUrl(key)}
                          onClick={this.onOpenTask}
                    >
                        <span className={'task-key-status-' + status.key}>
                            {key}
                        </span>
                        &nbsp;
                        <span className='task-status'>{status.display}</span>
                        &nbsp;
                        <span className='task-line-summary' title={summary}>
                            {summary}
                        </span>
                        {assignee.id &&
                        <>
                            &nbsp;
                            <span className='login'>{assignee.id}</span>
                        </>
                        }
                    </Link>
                    &nbsp;
                    {this.props.showWeight !== false && 'weight' in task &&
                    <span className='task-weight'>Вес: {task.weight}</span>
                    }
                    &nbsp;&nbsp;
                    {this.props.showTags && arrayIntersect(task.tags || [], this.props.showTags).map(t =>
                        <span className='tag' key={t}>{t}&nbsp;</span>
                    )}
                    &nbsp;&nbsp;&nbsp;
                    {this.props.canEdit !== false && !this.state.editMode &&
                    <Button view="clear" size="xs" onClick={this.onEditBegin}>
                        редактировать
                    </Button>
                    }
                    {/*<Button view="clear" size="xs" onClick={this.onClose}>*/}
                    {/*    Закрыть*/}
                    {/*</Button>*/}
                    {/*&nbsp;*/}
                    {document.location.hostname === 'local.adfox.yandex-team.ru' &&
                    <>
                        &nbsp;&nbsp;&nbsp;
                        <Button view="clear" size="xs" onClick={this.onDump}>
                            dump
                        </Button>
                    </>
                    }
                    {this.state.editMode &&
                    <TaskEditor
                        task={task}
                        allocationParents={this.props.allocationParents}
                        onSave={this.onEditSave}
                        onCancel={() => this.setState({editMode: false})}
                    />
                    }
                </span>
            </div>
        )
    }
}

TaskLine.propTypes = {
    mode: PropTypes.oneOf(['analyze', 'planning']),
    task: PropTypes.object.isRequired,
    showType: PropTypes.bool,
    showWeight: PropTypes.bool,
    canEdit: PropTypes.bool,
    onAssignTask: PropTypes.func,
    onEditSave: PropTypes.func,
};
