import React from 'react';
import {Spin} from '@yandex-lego/components/Spin/desktop/bundle';

import {apiCall, apiEditTask} from '../../startrek';

import './TaskStoreProvider.css';


export class TaskStoreProvider extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = {
            config: {},
            extendedConfig: {},
            commentId: null,
            version: -1,
            loading: true,
            updating: false,
            updateLocked: false,
            watch: false,
        };
    }

    updateConfigFromTask = async () => {
        if (this.state.updateLocked) {
            console.log('updateConfigFromTask: update locked');
            return;
        }

        this.setState({
            loading: true,
        });

        const comments = await apiCall(`issues/${this.props.taskKey}/comments`);

        for (const comment of comments) {
            const matches = comment.text.match(/%%((?:.|\n)*)%%/);
            if (matches) {
                const config = JSON.parse(matches[1]);
                let extendedConfig = JSON.parse(matches[1]);
                if (config && comment.version > this.state.version) {
                    console.log('new json config found');
                    if (this.props.middlewares) {
                        for (const middleware of this.props.middlewares) {
                            extendedConfig = await middleware(extendedConfig);
                        }
                    }

                    this.setState({
                        config,
                        extendedConfig,
                        commentId: comment.id,
                        loading: false,
                        version: comment.version,
                    });
                    return;
                }
            }
        }
        this.setState({
            config: {},
            extendedConfig: {},
            loading: false,
            version: 0,
        });
    };

    updateConfig = async (configPart) => {
        this.setState({
            updating: true,
        });

        let config = {...this.state.config, ...configPart};
        let extendedConfig = this.props.saveConverter ? this.props.saveConverter(config) : config;

        if (this.props.middlewares) {
            for (const middleware of this.props.middlewares) {
                extendedConfig = await middleware(extendedConfig);
            }
        }

        this.setState({
            extendedConfig,
            updating: false,
        });
    };

    componentDidMount = () => {
        this.updateConfigFromTask();
        if (this.props.watch) {
            this.timer = setInterval(this.updateConfigFromTask, 1000);
        }
    };

    componentWillUnmount() {
        if (this.timer) {
            clearInterval(this.timer);
            this.timer = null;
        }
    }

    setConfig = async (configPart, noMiddlewares) => {
        if (this.state.updateLocked) {
            console.log('setConfig: update locked');
            return;
        }
        console.log('setConfig');
        const config = {...this.state.config, ...configPart};
        this.setState({
            config,
            updateLocked: true,
        });

        const preparedConfig = this.props.saveConverter ? this.props.saveConverter(config) : config;

        const jsonConfig = JSON.stringify(preparedConfig, null, 4);
        const description = `Autogenerated by planner\n\n%%\n${jsonConfig}\n%%\n`;

        let comment;
        if (this.state.commentId) {
            const uri = `issues/${this.props.taskKey}/comments/${this.state.commentId}`;
            comment = await apiCall(uri, {text: description}, 'PATCH');
        } else {
            const uri = `issues/${this.props.taskKey}/comments`;
            comment = await apiCall(uri, {text: description}, 'POST');
        }
        if (config.name) {
            await apiEditTask(this.props.taskKey, {summary: config.name});
        }
        console.log('new config task was saved');

        let extendedConfig = {...config};
        if (this.props.middlewares && !noMiddlewares) {
            for (const middleware of this.props.middlewares) {
                extendedConfig = await middleware(extendedConfig);
            }
        }

        this.setState({
            config,
            extendedConfig,
            version: comment.version,
            updateLocked: false,
            commentId: comment.id,
        });
    };

    render() {
        if (this.state.loading && !this.props.watch) {
            return (
                <div className={'task-store-provider'}>
                    <Spin
                        progress
                        className={'task-store-provider__spinner'}
                        view='default'
                        size='m'
                    />
                </div>
            );
        }

        return this.state.version >= 0 ?
            this.props.children(
                this.state.extendedConfig,
                this.setConfig,
                this.updateConfig,
                {
                    saveInProgress: this.state.updateLocked,
                    updating: this.state.updating,
                }) :
            null;
    }
}
