import React, {Component} from 'react';
import Links from './links';
import './tree.css';
import {StartrekClient, buildTree, budgetFromTree, nodeFormatter, budgetFormatter} from './exporter';
import HTMLReactParser from 'html-react-parser';

const tracker = new StartrekClient();
const trackerTicketMask = '^[a-zA-Z]+-[0-9]+$';
const trackerLink = 'https://st.yandex-team.ru/';
const findGetParameter = (param) => {
    let result = '',
        parsed = [];
    let getString = window.location.search.substr(1);
    getString = getString.replace(/[;'"]/, '');
    let parameters = getString.split('&');
    for (let i = 0; i < parameters.length; i++) {
        parsed = parameters[i].split('=');
        if (parsed[0].toLowerCase() === param.toLowerCase()) {
            result = decodeURIComponent(parsed[1]);
        }
    }
    return result;
};

let ticket = findGetParameter('ticket').toUpperCase();
let budgetPreset = findGetParameter('budgetPreset') || 'none';
let formatter = findGetParameter('formatter') || 'markdown2';
let hideCompleted = findGetParameter('hideCompleted') || 'no';

let isLocalhost = window.location.hostname === 'localhost';

const replaceInitialSpaces = (text) => {
    let index = text.indexOf('.') - 1;
    if (index < 1) index = text.indexOf('*') - 1;
    if (index < 1) return text;
    else {
        let prefix = '',
            space = '\u00A0';
        for (let i = 0; i < index - 1; i++) prefix += space;
        return `${prefix}${text}`;
        // return `<span style="margin-left: ${index}0px">${text}</span> `;
    }
};

const formatMarkdownTreeForHTML = (text, hideCompleted) => {
    return (
        text
            // .replace(/(^|\n)===(.)+\n/g, '<h3>$&</h3>')
            .replace(/(^|===|Показатели бюджета)(.)+\n/g, '<h3>$&</h3>')
            // .replace(/\n.+\n/g, '<p>$&</p>')
            .replace(/( )*(1\..*--|\d\..*\[Закрыт\]|\*.*--)\n/g, `${hideCompleted === 'yes' ? '' : '$&'}`)
            .replace(/[a-zA-Z]{2,}-[0-9]+/g, `<a href='${trackerLink}$&' target=_blank>$&</a>`)
            .replace(
                /([\d]?.?[\d]sp)(.*)(\[sprint.*\])(.*)\n/g,
                `<span class="darkGreen">$1</span>$2<span class="darkGreen">$3</span>$4\n`,
            )
            .replace(/ +(\d+.)+[+ ]/g, replaceInitialSpaces)
            .replace(/ +\*/g, replaceInitialSpaces)
            .replace(/ (\d+[.])+([+]? )/g, '<span class="lightgray">$&</span>')
            .replace(/ +\* /g, '<span class="lightgray">$&</span>')
            .replace(/\*\*.*\*\*/g, '<span class="summaryTask">$&</span>')
            .replace(/(--.*--|\[Закрыт\])/g, `<span class="closedTask">$&</span>`)
            .replace(/\n/g, '<br />')
    );
};

class Tree extends Component {
    constructor(props) {
        super(props);
        this.state = {
            ticket,
            links: [],
            budgetPreset,
            formatter,
            hideCompleted,
        };
        this.handleChange = this.handleChange.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
        this.handleSelectBudgetPreset = this.handleSelectBudgetPreset.bind(this);
        this.handleSelectFormatter = this.handleSelectFormatter.bind(this);
        this.handleSelectHideCompleted = this.handleSelectHideCompleted.bind(this);
    }

    getNodeText(node, formatter, hideCompleted) {
        const formatNode = nodeFormatter(formatter);
        return formatMarkdownTreeForHTML(formatNode(node), hideCompleted || this.state.hideCompleted);
    }

    getNodeBudget(node, formatter, preset) {
        const formatBudget = budgetFormatter(formatter);
        const budget = budgetFromTree(node, preset);
        const budgetText = formatMarkdownTreeForHTML(formatBudget(budget));
        return {budget, budgetText};
    }

    handleChange(event) {
        this.setState({ticket: event.target.value});
    }

    handleSelectBudgetPreset(event) {
        budgetPreset = event.target.value;
        if (this.state.tree) {
            let budget, budgetText;
            budget = budgetFromTree(this.state.tree, budgetPreset);
            const formatBudget = budgetFormatter(this.state.formatter);
            if (budget) budgetText = formatMarkdownTreeForHTML(formatBudget(budget));
            this.setState({budgetPreset, budget, budgetText});
        } else this.setState({budgetPreset});
    }

    async handleSelectFormatter(event) {
        let formatter = event.target.value;
        if (this.state.tree) {
            const nodeText = this.getNodeText(this.state.tree, formatter);
            if (this.state.budgetPreset !== 'none') {
                const {budget, budgetText} = this.getNodeBudget(this.state.tree, formatter, this.state.budgetPreset);
                this.setState({formatter, budget, nodeText, budgetText});
            } else {
                this.setState({formatter, nodeText});
            }
        } else this.setState({formatter});
    }

    async handleSelectHideCompleted(event) {
        let hideCompleted = event.target.value;
        if (this.state.tree) {
            const nodeText = this.getNodeText(this.state.tree, this.state.formatter, hideCompleted);
            this.setState({hideCompleted, nodeText});
        } else this.setState({hideCompleted});
    }

    async handleSubmit(event) {
        this.setState({ticket: ('' + this.state.ticket).toUpperCase()});
        // event.preventDefault();
    }

    async getFirstLevelLinks(ticket) {
        if (ticket === '' || !new RegExp(trackerTicketMask).test(ticket)) return;
        const issue = await tracker.get(ticket);
        const links = await tracker.getLinks(ticket);
        if (links) {
            let filteredLinks = links.filter((link) => link.type.id === 'subtask' && link.direction === 'outward');
            const firstLevelTree = {
                id: ticket,
                title: issue.summary,
                parent: '',
                children: filteredLinks.map((link) => {
                    return {
                        id: link.object.key,
                        title: link.object.display,
                        parent: ticket,
                        children: [],
                        isFinished: link.status.key === 'closed',
                    };
                }),
            };
            const nodeText = this.getNodeText(firstLevelTree, this.state.formatter);
            this.setState({nodeText});
        }
    }

    async getTree(ticket) {
        let tree = await buildTree(ticket);
        const nodeText = this.getNodeText(tree, this.state.formatter);
        if (this.state.budgetPreset !== 'none') {
            const {budget, budgetText} = this.getNodeBudget(tree, this.state.formatter, this.state.budgetPreset);
            this.setState({tree, budget, nodeText, budgetText});
        } else {
            this.setState({tree, nodeText});
        }
    }

    componentDidMount() {
        // document.getElementById('ticket').focus();
        if (new RegExp(trackerTicketMask).test(ticket)) {
            this.getFirstLevelLinks(ticket);
            this.getTree(ticket);
        }
    }

    componentWillUnmount() {
        ticket = '';
    }

    render() {
        return (
            <div>
                <div id="menuWrapper">
                    <div id="menu">
                        <h1>Дерево задач {isLocalhost ? <span className="red">offline</span> : ''}</h1>
                        <form onSubmit={this.handleSubmit}>
                            <div className="form_main_input">
                                <label>
                                    Ticket:{' '}
                                    <input
                                        type="text"
                                        name="ticket"
                                        id="ticket"
                                        value={this.state.ticket}
                                        onChange={this.handleChange}
                                    />
                                    <input type="submit" value="build tree" />
                                </label>
                            </div>

                            <div
                                className="form_radio_group"
                                value={this.state.formatter}
                                onChange={this.handleSelectFormatter}
                            >
                                <div className="form_radio_group-item">
                                    <input
                                        id="formatter-text"
                                        type="radio"
                                        name="formatter"
                                        value="text"
                                        checked={this.state.formatter === 'text'}
                                    />
                                    <label htmlFor="formatter-text">text</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="formatter-csv"
                                        type="radio"
                                        name="formatter"
                                        value="csv"
                                        checked={this.state.formatter === 'csv'}
                                    />
                                    <label htmlFor="formatter-csv">csv</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="formatter-markdown"
                                        type="radio"
                                        name="formatter"
                                        value="markdown"
                                        checked={this.state.formatter === 'markdown'}
                                    />
                                    <label htmlFor="formatter-markdown">markdown</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="formatter-markdown2"
                                        type="radio"
                                        name="formatter"
                                        value="markdown2"
                                        checked={this.state.formatter === 'markdown2'}
                                    />
                                    <label htmlFor="formatter-markdown2">markdown2</label>
                                </div>
                            </div>

                            <div
                                className="form_radio_group"
                                value={this.state.budgetPreset}
                                onChange={this.handleSelectBudgetPreset}
                            >
                                <div className="form_radio_group-item">
                                    <input
                                        id="preset-none"
                                        type="radio"
                                        name="budgetPreset"
                                        value="none"
                                        checked={this.state.budgetPreset === 'none'}
                                    />
                                    <label htmlFor="preset-none">hide budget</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="preset-all"
                                        type="radio"
                                        name="budgetPreset"
                                        value="all"
                                        checked={this.state.budgetPreset === 'all'}
                                    />
                                    <label htmlFor="preset-all">all</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="preset-ppd"
                                        type="radio"
                                        name="budgetPreset"
                                        value="ppd"
                                        checked={this.state.budgetPreset === 'ppd'}
                                    />
                                    <label htmlFor="preset-ppd">ppd</label>
                                </div>
                            </div>

                            <div
                                className="form_radio_group"
                                value={this.state.hideCompleted}
                                onChange={this.handleSelectHideCompleted}
                            >
                                <div className="form_radio_group-item">
                                    <input
                                        id="hide-yes"
                                        type="radio"
                                        name="hideCompleted"
                                        value="yes"
                                        checked={this.state.hideCompleted === 'yes'}
                                    />
                                    <label htmlFor="hide-yes">hide completed</label>
                                </div>
                                <div className="form_radio_group-item">
                                    <input
                                        id="hide-no"
                                        type="radio"
                                        name="hideCompleted"
                                        value="no"
                                        checked={this.state.hideCompleted === 'no'}
                                    />
                                    <label htmlFor="hide-no">show</label>
                                </div>
                            </div>
                        </form>
                    </div>
                    <div id="scrollBar"></div>
                </div>

                <div id="main2">
                    <Links ticket={ticket} tree={this.state.tree} />
                    <div>{this.state.nodeText && HTMLReactParser(this.state.nodeText)}</div>
                    <div>{this.state.budgetText && HTMLReactParser(this.state.budgetText)}</div>
                </div>
            </div>
        );
    }
}

export default Tree;
