/******************************
 * Название: tracker wbs client
 * Описание: Строит Иерархическую структуру тикетов по типу "подзадача".
 *           Требует прописать в лежащий на один уровень каталогов выше secret.txt OAUTH ключ:
 *           Для получения OAuth-токена зайдите из под нужного пользователя на
 *             https://oauth.yandex-team.ru/authorize?response_type=token&client_id=5f671d781aca402ab7460fde4050267b
 * Авторы: copytenz, tzota, март-июнь 2020
 *******************************/

// Внешние модули
import {TreeNode, BudgetClass} from './modules/types';
import StartrekClient from './modules/StartrekClient';
import {budgetFromTree} from './modules/Budget';
import {buildTree} from './modules/treeBuilder';
import {formatBudgetNumber, updateTextWithAddition} from './modules/commonFunctions';
import {nodeFormatter, budgetFormatter} from './modules/formatters/selectFormatter';

// Настройки
import {settings} from './modules/infra/settings';

const moduleDescription = `
tracker-wbs-client

USAGE: npm start ROOT_TICKET [TARGET_TICKET]

ARGUMENTS:
* If started without arguments - shows this message
* If ROOT_TICKET is set - tries to build structure
* If TARGET_TICKET is set - tries to write results to target ticket otherwise prints results to standard output

EXAMPLES:
  $ npm start PPD-1 # will print results to standard output
  $ npm start PPD-1 PPD-10 # will save results to startrek ticket with id PPD-10
`;

// Начальные переменные
const ticket = process.argv[2] || ''; // Номер корневого тикета от которого строится дерево
let targetTicketId = ''; // Идентификатор тикета куда нужно записать результат
const trackerTicketMask = '^[a-zA-Z]+-[0-9]+$'; // Маска номера тикета для проверки параметров командной строки
const START_TOKEN = '=== Иерархическая структура работ\n';
const END_TOKEN = '=====\n';
if (!ticket || !new RegExp(trackerTicketMask).test(ticket)) {
    console.log(moduleDescription);
    process.exit();
}

targetTicketId = new RegExp(trackerTicketMask).test(process.argv[3]) ? process.argv[3] : '';

/**
 * Функция записывает результат форматирования дерева в тикет
 * @param {string} targetTicket номер тикета, в который нужно записать текст
 * @param {object} updateObject объект с деталями, которые нужно записать/обновить
 * @returns {string} Текст ошибки, в случае ошибки
 */
const writeSomethingToTicket = async (targetTicket: string, updateObject: any): Promise<string> => {
    if (!targetTicket)
        return `* Error in writeSomethingToTicket with target ${targetTicket}:\n Target ticket ID is not set - ignoring writing to ticket`;
    try {
        const tracker = new StartrekClient();
        const queryResult = await tracker.update(targetTicket, updateObject);
        if (queryResult === '200: OK') return 'Result is written to: https://st.yandex-team.ru/' + targetTicket;
        else return queryResult;
    } catch (err: any) {
        return `Error in writeWBSToTicket with target ${targetTicket}:\n ${err.name} ${err.message}`;
    }
};

/**
 * Функция записывает результат форматирования дерева в тикет
 * @param {string} targetTicket номер тикета, в который нужно записать текст
 * @param {string} wbsText Текст, который нужно записать
 * @returns {string} Текст ошибки, в случае ошибки
 */
const writeWBSToTicket = async (targetTicket: string, wbsText: string): Promise<string> => {
    if (!targetTicket)
        return `* Error in writeWBSToTicket with target ${targetTicket}:\n Target ticket ID is not set - ignoring writing to ticket`;
    try {
        const tracker = new StartrekClient();
        const target = await tracker.get(targetTicket);
        const wbsTextObject = {
            description: updateTextWithAddition(target.description, wbsText, START_TOKEN, END_TOKEN),
        };
        return writeSomethingToTicket(targetTicket, wbsTextObject);
    } catch (err: any) {
        return `Error in writeWBSToTicket with target ${targetTicket}:\n ${err.name} ${err.message}`;
    }
};

const writeSpentSpToParentTicket = async (targetTicket: string, spentSp: string, totalSP: string): Promise<string> =>
    await writeSomethingToTicket(targetTicket, {
        spentSp,
        storyPoints: totalSP,
    });

// Основная функция
export default async function TrackerWbsBuilder(ticketId: string): Promise<void> {
    let result;
    const tree: TreeNode = await buildTree(ticketId);
    let budgetText = '';
    if (!settings.formatter) settings.formatter = 'markdown';
    const formatBudget: Function = budgetFormatter(settings.formatter);
    if (settings.budgetPreset) {
        const budget: BudgetClass | undefined = budgetFromTree(tree, settings.budgetPreset);
        if (budget) budgetText = formatBudget(budget);
        if (budget && settings.updateSpentSpToParentTicket)
            result = await writeSpentSpToParentTicket(
                ticketId,
                formatBudgetNumber(budget.totalSpentSP),
                formatBudgetNumber(budget.totalSpentSP + budget.totalEstimatedSP),
            );
        console.log(result);
    }
    const formatNode: Function = nodeFormatter(settings.formatter);
    result = '' + formatNode(tree) + budgetText;
    if (targetTicketId) result = await writeWBSToTicket(targetTicketId, result);
    console.log(result);
}

TrackerWbsBuilder(ticket);
