import {TreeNode, BudgetTeamDefinition, BudgetTeamItem, BudgetClass} from './types';
import {budgetSettings, budgetResolutionsExclude} from '../budgetSettings';
import {logError, stringContainsOneOf, stringDoesntContainAnyOf, arrayContainsOneOf} from './commonFunctions';

export default class Budget implements BudgetClass {
    teamFilter: string[];
    excludeTags: string[];
    teams: BudgetTeamItem[];
    teamDefinitions: BudgetTeamDefinition[];
    originalSP: number;
    totalSpentSP: number;
    totalEstimatedSP: number;

    constructor(
        teamFilter: string[],
        teamDefinitions: BudgetTeamDefinition[],
        excludeTags: string[],
        originalSP: number,
    ) {
        this.teamFilter = teamFilter;
        this.excludeTags = excludeTags.length > 0 ? excludeTags : [];
        this.teamDefinitions = teamDefinitions;
        this.teams = [];
        this.originalSP = originalSP;
        this.totalSpentSP = 0;
        this.totalEstimatedSP = 0;
    }

    addOrUpdate(team: string | BudgetTeamItem): void {
        if (typeof team === 'string') {
            if (this.indexOf(team) < 0)
                this.teams.push({
                    name: team,
                    spentSP: 0,
                    estimatedSP: 0,
                    numEndTasksWithoutEstimation: 0,
                    tags: [],
                });
        } else if (typeof team === 'object') {
            const id = this.indexOf(team.name);
            if (id < 0) {
                this.teams.push(team);
            } else {
                Object.assign(this.teams[id], team);
            }
        } else {
            throw new Error('* ERROR: Team for object Budget is not defined and node can not be built');
        }
    }

    indexOf(value: string): number {
        if (!this.teams.length) return -1;
        for (let i = 0; i < this.teams.length; i++) {
            if (this.teams[i].name === value) return i;
        }
        return -1;
    }

    defineTeam(node: TreeNode): string {
        if (!(typeof node === 'object' && node.id && typeof node.id === 'string'))
            logError('Can not define team, node is not of correct object type');

        const queue = `${node.id}`.substring(0, node.id.indexOf('-', 0)).toUpperCase();
        if (!queue) return node.id;
        const nodeTitle = node.title || '';
        if (!node.tags) node.tags = [];
        for (const i in this.teamDefinitions) {
            const definedTeam = this.teamDefinitions[i];
            if (
                definedTeam.queues.includes(queue) &&
                arrayContainsOneOf(node.tags, definedTeam.tags) &&
                stringContainsOneOf(nodeTitle, definedTeam.titleContains) &&
                stringDoesntContainAnyOf(nodeTitle, definedTeam.titleDoesntContain)
            ) {
                return definedTeam.name;
            }
        }
        return queue;
    }

    calcBudgetRecursively(node: TreeNode): void {
        const teamName = this.defineTeam(node);
        if (this.indexOf(teamName) < 0) this.addOrUpdate(teamName);
        const id = this.indexOf(teamName);
        const team = this.teams[id];
        const printTeamBudget = this.teamFilter.length === 0 || this.teamFilter.includes(team.name);
        const nodeTags = node.tags || [];
        const nodeResolution = node.resolution?.key ? '' + node.resolution.key : '';
        if (
            printTeamBudget &&
            !arrayContainsOneOf(nodeTags, this.excludeTags) &&
            budgetResolutionsExclude.indexOf(nodeResolution) === -1
        ) {
            if (node.storyPoints) {
                if (node.isFinished) {
                    team.spentSP = (team.spentSP || 0) + node.storyPoints;
                } else {
                    team.estimatedSP = (team.estimatedSP || 0) + node.storyPoints;
                }
            } else {
                if (!node.isFinished && (!node.children || node.children.length === 0)) {
                    team.numEndTasksWithoutEstimation = (team.numEndTasksWithoutEstimation || 0) + 1;
                }
            }
            this.addOrUpdate(team);
        }
        if (node.children) {
            for (let i = 0; i < node.children.length; i++) {
                this.calcBudgetRecursively(node.children[i]);
            }
        }
    }

    // Функция считает бюджет по группам разработки
    calcBudget = (tree: TreeNode): void => {
        this.calcBudgetRecursively(tree);
        let team;
        for (let i = 0; i < this.teams.length; i++) {
            team = this.teams[i];
            if (team.spentSP) this.totalSpentSP += team.spentSP;
            if (team.estimatedSP) this.totalEstimatedSP += team.estimatedSP;
        }
    };
}

// Функция расчитывает бюджет и возвращает его отформатированным
export const budgetFromTree = (
    tree: TreeNode,
    preset: string,
    formatter?: string,
    excludeTags?: string[],
): BudgetClass | undefined => {
    preset = `${preset}`.toLowerCase();
    if (!preset || preset === 'none' || !budgetSettings[preset]) return undefined;
    if (!formatter) formatter = 'markdown';

    excludeTags = excludeTags && excludeTags.length ? excludeTags : budgetSettings[preset].excludeFromBudgetTags;

    const originalSP: number = tree && tree.originalStoryPoints ? tree.originalStoryPoints : 0;
    const budget = new Budget(
        budgetSettings[preset].teamFilterForBudget,
        budgetSettings[preset].teamDefinitions,
        excludeTags || [],
        originalSP,
    );
    budget.calcBudget(tree);
    return budget;
};
