import {TreeNode} from '../types';
import {
    isFinished,
    hasChildren,
    decorateWith,
    noop,
    add,
    addInitialSpace,
    compareNodesForSort,
} from './formattersCommon';
import {compose, concat} from 'ramda';

/** закрытие тикеты с детьми сворачивает */
const collapseNode = concat('');

/** @param {TreeNode} node */
const markFinished = (node: TreeNode): any => (isFinished(node) ? decorateWith('') : noop);

/** @param {TreeNode} node  */
const markHasChildren = (node: TreeNode): any => (hasChildren(node) ? decorateWith('') : noop);

/** @param {TreeNode} node  */
const markFinishedSubtree = (node: TreeNode): any => (isFinished(node) && hasChildren(node) ? collapseNode : noop);

/** @param {TreeNode} node  */
const addSprint = (node: TreeNode): any => {
    if (node.sprint && node.sprint.length) {
        let result = '';
        for (let k = 0; k < node.sprint.length; k++) {
            if (k > 0) result += ', ';
            result += node.sprint[k].display;
        }
        return add(' [sprint:' + result + ']');
    } else return noop;
};
/** @param {TreeNode} node  */
const addSP = (node: TreeNode): any => (node.storyPoints ? add(' [' + node.storyPoints + 'SP]') : noop);

/** @param {TreeNode} node  */
const addStatus = (node: TreeNode): any =>
    node.status && node.status.display ? add(' [' + node.status.display + ']') : noop;

/** @param {TreeNode} node  */
const addTitle = (node: TreeNode): any => (node.title ? add(': ' + node.title) : noop);

const getKey = (node: TreeNode): string => node.id;

/**
 * Вывод узла в виде markdown
 * @param {TreeNode} node
 * @returns string
 */
export const getLineForNode = (node: TreeNode): string => {
    let result = '';
    const decorate = compose(
        markFinished(node),
        markHasChildren(node),
        addSprint(node),
        addSP(node),
        addStatus(node),
        addTitle(node),
    );
    //@ts-ignore
    result += decorate(getKey(node));
    result = addInitialSpace(result);
    result = markFinishedSubtree(node)(result);
    return result;
};

const getPrefix = (prefix: string, i?: number): string => `${prefix}${(i || 0) + 1}.`;

/**
 * Вывод узла и его детей в виде markdown
 * @param {TreeNode} node
 * @param {string} incomingPrefix
 * @returns string
 */
const formatRecursively = (node: TreeNode, incomingPrefix: string): string => {
    let result = '';
    if (node.children && node.children.length) {
        const prefix = '  ' + incomingPrefix;
        const children = node.children.sort((a: TreeNode, b: TreeNode) => compareNodesForSort(a, b));
        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            const tmpPrefix = getPrefix(prefix, i);
            result += tmpPrefix + getLineForNode(child) + '\n';
            if (child.children && child.children.length > 0) {
                result += formatRecursively(child, tmpPrefix);
            }
        }
    }
    return result;
};

/**
 * Компиляция полного текста в виде markdown
 * @param {TreeNode} node
 * @returns string
 */
export const nodeToText = (node: TreeNode): any => {
    return `${node.id} ${node.title}\n${formatRecursively(node, '')}`;
};
