import { useQuery, useMutation, queryCache } from 'react-query';
import { api, apiWithHeaders } from './api';
function commentInputDataAdapter(comment) {
    try {
        // Новый формат данных, где мета-информация хранится под спойлером.
        // Пример — https://regex101.com/r/cIciwP/1/
        let matched = comment.text.match(/([\s\S]+)(?:<{\n?(.+)\n?}>)/);
        if (matched !== null) {
            const [, content, data] = matched;
            return { ...comment, data: { content, ...JSON.parse(data) } };
        }
        // Старый формат данных, где все хранилось в виде plain-текста.
        // Пример — https://regex101.com/r/j9K45g/2
        matched = comment.text.match(/(.+)\n(?:retroboard-created-task:(.+)\n)?([\s\S]+)/);
        if (matched !== null) {
            const [, type, task, content] = matched;
            return { ...comment, data: { type, content, task, resolved: false, login: undefined } };
        }
        throw new Error('');
    }
    catch (_e) {
        return { ...comment, data: { type: '', content: '', resolved: false, login: undefined } };
    }
}
function commentOutputDataAdapter(data) {
    const { content, ...rest } = data;
    return `${content}<{\n${JSON.stringify(rest)}\n}>`;
}
const updateItemInCollection = (collection, item) => {
    return collection.map(cItem => {
        if (cItem.id === item.id) {
            return { ...cItem, ...item };
        }
        return cItem;
    });
};
const addItemToCollection = (collection, item) => {
    return [...collection, item];
};
const removeItemFromCollection = (collection, item) => {
    return collection.filter(c => c.id !== item.id);
};
const getLastId = ({ data }) => {
    return data?.[data.length - 1]?.id;
};
export const getList = async (taskId) => {
    let firstPage = await apiWithHeaders(`issues/${taskId}/comments`);
    const allComments = [...firstPage.data];
    let lastId = getLastId(firstPage);
    while (lastId && allComments.length !== firstPage.headers.totalCount) {
        const nextPage = await apiWithHeaders(`issues/${taskId}/comments?id=${lastId}`);
        allComments.push(...nextPage.data);
        lastId = nextPage.data.length ? getLastId(nextPage) : null;
    }
    return allComments.map(commentInputDataAdapter);
};
export const useList = (taskId) => {
    return useQuery('comments', () => getList(taskId), {
        refetchInterval: 5000,
    });
};
export const Vote = async ({ taskId, commentId, type = 'vote' }) => {
    const method = type === 'vote' ? 'POST' : 'DELETE';
    const data = await api(`issues/${taskId}/comments/${commentId}/reactions/LIKE`, { method });
    return data;
};
export const useVote = () => {
    return useMutation(Vote, {
        onMutate: ({ commentId, type }) => {
            queryCache.cancelQueries('comments');
            const previousComments = queryCache.getQueryData('comments');
            queryCache.setQueryData('comments', (comments) => {
                const current = comments.find(c => c.id === commentId);
                const reactions = current?.reactionsCount || { like: 0 };
                const newData = {
                    ...current,
                    reactionsCount: {
                        ...reactions,
                        like: reactions.like + (type === 'vote' ? 1 : -1),
                    },
                    ownReactions: type === 'vote' ? ['like'] : null,
                };
                return updateItemInCollection(comments, newData);
            });
            return () => queryCache.setQueryData('comments', previousComments);
        },
        onSuccess: data => {
            queryCache.setQueryData('comments', oldComments => {
                return updateItemInCollection(oldComments, data);
            });
        },
    });
};
export const addComment = async ({ taskId, data }) => {
    return await api(`issues/${taskId}/comments`, {
        method: 'POST',
        body: JSON.stringify({ text: commentOutputDataAdapter(data) }),
    });
};
export const useAddComment = () => {
    return useMutation(addComment, {
        onSuccess: comment => {
            queryCache.setQueryData('comments', comments => addItemToCollection(comments, commentInputDataAdapter(comment)));
        },
    });
};
export const editComment = async ({ taskId, commentId, data }) => {
    return await api(`issues/${taskId}/comments/${commentId}`, {
        method: 'PATCH',
        body: JSON.stringify({ text: commentOutputDataAdapter(data) }),
    });
};
export const useEditComment = () => {
    return useMutation(editComment, {
        onMutate: ({ commentId, data }) => {
            queryCache.cancelQueries('comments');
            const previousComments = queryCache.getQueryData('comments');
            queryCache.setQueryData('comments', comments => updateItemInCollection(comments, { id: commentId, data }));
            return () => queryCache.setQueryData('comments', previousComments);
        },
        onSuccess: comment => {
            queryCache.setQueryData('comments', comments => updateItemInCollection(comments, commentInputDataAdapter(comment)));
        },
    });
};
export const removeComment = async ({ taskId, commentId }) => {
    const method = 'DELETE';
    try {
        await api(`issues/${taskId}/comments/${commentId}`, {
            method,
        });
    }
    catch (e) { }
    return null;
};
export const useRemoveComment = () => {
    return useMutation(removeComment, {
        onMutate: ({ commentId }) => {
            queryCache.cancelQueries('comments');
            const previousComments = queryCache.getQueryData('comments');
            queryCache.setQueryData('comments', comments => {
                return removeItemFromCollection(comments, { id: commentId });
            });
            return () => queryCache.setQueryData('comments', previousComments);
        },
        onSuccess: (_, { commentId }) => {
            queryCache.setQueryData('comments', oldComments => {
                return removeItemFromCollection(oldComments, { id: commentId });
            });
        },
    });
};
