#!/usr/bin/perl

=encoding UTF-8

=cut

=head1 DESCRIPTION

Проверяет SLA в открытых задачах

=head1 PARAMS

    tg_chanel - для отладки писать только в этот канал (например -354212043)
    no-send   - вместо отправки писать в STDOUT

=head1 USAGE


=begin comment

Команды:


=cut

# common modules
use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use lib::abs;

use Carp;
use Data::Dumper;
use File::Slurp;
use Getopt::Long;
use HTTP::Request::Common qw( GET POST PUT );
use HTTP::Request;
use HTTP::Tiny;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use JSON::PP;
use LWP::UserAgent;
use List::Util qw(first);
use Net::INET6Glue;
use Pod::Usage;
use Sys::RunAlone silent => 1;
use XML::Simple;
use Yandex::StarTrek;
use Time::Local;

my $chanel_target;
$chanel_target = {
    ADFOX   => {
        id    => '-1001146269948',
        title => 'Дежурства Adfox',
        name  => 'ADFOX',
        limit => 500,
        check => sub {$_[0]{ticket}{queue}{key} eq 'ADFOX'},
    },
    frontend => {
        id    => '-378824775',
        title => 'Аварийный чат фронта ПИ',
        name  => 'frontend',
        limit => 500,
        check => sub {!$chanel_target->{ADFOX}{check}(@_) and grep {$_ eq 'frontend'} @{$_[0]{ticket}{tags} // []}},
    },
    backend => {
        id    => '-1001446734384',
        title => 'ПИ Дежурные (бэкенд)',
        name  => 'backend',
        limit => 100,
        check => sub {not($chanel_target->{ADFOX}{check}(@_) or $chanel_target->{frontend}{check}(@_))},
    },
    support => {
        id    => '-318756299',
        title => 'Поддержка ПИ',
        name  => 'support',
        limit => 500,
        check => undef,
    },
    full_list    => {
        id     => '-1001207666739',
        title  => 'test_chanel_pi',
        name   => 'full_list',
        limit  => 0,
        check  => undef,
        failed => 1,
    }
};

my $PI_SECRETS;
my $BROWSER;

my $DEBUG_TG_CHANNEL = '';
my $REAL_SEND = 1;

sub LOG ($) {_log('LOG', shift)}
sub LOGF ($@) {_logf('LOG', @_)}
sub INFO ($) {_log('INFO', shift)}
sub INFOF ($@) {_logf('INFO', @_)}
sub WARN ($) {_log('WARN', shift)}
sub WARNF ($@) {_logf('WARN', @_)}
sub ERROR ($) {_log('ERROR', shift)}
sub ERRORF ($@) {_logf('ERROR', @_)}

my @CID_TO_SLA = (
    [20   => {warn => [538, 0],    fail => 540}],
    [100  => {warn => [250, 0],    fail => 252}],
    [500  => {warn => [46, 24, 0], fail => 48}],
    [1000 => {warn => [10, 6,  0], fail => 12}],
);

my $FIELD_FOR_CHECK = 'weight';
main();

sub main {

    get_args();

    if ($DEBUG_TG_CHANNEL) {
        $_->{id} = $DEBUG_TG_CHANNEL for values %$chanel_target;
    }

    LOG('START');
    eval {process_tickets()};
    if ($@) {
        ERROR $@;
        die 'FAIL';
    }
    LOG('END');
}

sub need_alert {
    my ($start, $cid) = @_;

    my $spend = int(time/3600) - int($start/3600);
    my $alerts;
    for my $r (@CID_TO_SLA) {
        if ($cid <= $r->[0]) {
            my $alert;
            if ($spend > $r->[1]{fail}) {
                $alert = $r->[1]{fail} - $spend;
            } else {
                for my $d (@{$r->[1]{warn}}) {
                    if ($spend >= $d) {
                        $alert = $d;
                        last;
                    }
                }
            }
            $alerts = {cid => $r->[0], alert => $alert} if defined $alert;
            last;
        }
    }

    return $alerts;
}

sub process_tickets {

    my $startrek_tickets =
      get_filtred_startrek_tickets([qw(ADFOX PI)], [qw(closed resolved later needInfo draft merged)]);
    my $data = get_maked_data();

    my @need_alert;
    my %new_data;
    for my $ticket (@$startrek_tickets) {
        next unless $ticket->{$FIELD_FOR_CHECK} and $ticket->{$FIELD_FOR_CHECK} =~ /^-?\d+$/ and $ticket->{$FIELD_FOR_CHECK} >= 100;

        my $t_data = $data->{$ticket->{key}} // {};
        my $start = get_time($ticket->{createdAt});
        if (my $alert = need_alert($start, $ticket->{$FIELD_FOR_CHECK})) {
            unless (exists $t_data->{$alert->{cid}}{$alert->{alert}}) {
                push @need_alert,
                  {
                      key    => $ticket->{key},
                      alert  => $alert->{alert},
                      hours  => abs($alert->{alert}),
                      status => ($alert->{alert} < 0 ? 'Expired' : $alert->{alert} ? 'Waiting' : 'New'),
                      active => ($alert->{alert} < 0 ? 0 : 1),
                      ticket => $ticket,
                  };
                $t_data->{$alert->{cid}}{$alert->{alert}} = undef;
            }
        }
        $new_data{$ticket->{key}} = $t_data;
    }

    my %to_send;
    for my $alert (@need_alert) {
        prepare_message(\%to_send, $alert);
        LOG("ALERT:\t$alert->{chanel}\t$alert->{ticket}{$FIELD_FOR_CHECK}\t$alert->{key}\t$alert->{alert}");
    }

    for my $chanel (keys %to_send) {
        send_messages($chanel, $to_send{$chanel});
    }

    set_maked_data(\%new_data);
}

sub prepare_message {
    my ($queue, $alert) = @_;

    for my $key (keys %$chanel_target) {
        my $push =
            ($alert->{ticket}{$FIELD_FOR_CHECK} >= $chanel_target->{$key}{limit})
            &&
            ($alert->{active} or $chanel_target->{$key}{failed});

        if ($chanel_target->{$key}{check}) {
            if ($chanel_target->{$key}{check}($alert)) {
                $alert->{chanel} = $chanel_target->{$key}{name};
                push @{$queue->{$key}}, $alert if $push;
            }
        } else {
            push @{$queue->{$key}}, $alert if $push;
        }
    }
}

sub send_messages {
    my ($chanel, $list) = @_;

    my %message;
    for my $alert (sort {$b->{ticket}{$FIELD_FOR_CHECK} <=> $a->{ticket}{$FIELD_FOR_CHECK} or $b->{hours} <=> $a->{hours}} @$list) {
        my $description = $alert->{ticket}{summary};
        $description =~ s/&/&amp;/g;
        $description =~ s/</&lt;/g;
        $description =~ s/>/&gt;/g;
        $description =~ s/"/&quot;/g;

        push @{$message{$alert->{status}}{$alert->{chanel}}},
          sprintf(
            "<code>%7d%8d%14s %10s</code> <a href=\"https://st.yandex-team.ru/%s\">%s</a>\n",
            $alert->{ticket}{$FIELD_FOR_CHECK},
            $alert->{hours}, 
            $alert->{key}, 
            ($alert->{ticket}{assignee} ? $alert->{ticket}{assignee}{id} : '-'),
            $alert->{key}, 
            $description,
          );
    }

    my $token = get_secret('telegram_release')->{token};
    my $title = sprintf "<code>%7s%8s%14s %10s %s</code>", $FIELD_FOR_CHECK, 'hours', 'ticket', 'assignee', 'description';
    for my $type (keys %message) {
        for my $ch (keys %{$message{$type}}) {
            my $mlist = $message{$type}{$ch};
            my $cnt   = @$mlist;
            my $page  = 0;
            my $str   = '';
            my @to_send;
            while ($str or @$mlist) {
                $page++;
                my $message = "<b>$type $ch</b>: <TASKS>/$cnt (page $page/<PAGES>)\n$title\n";
                my $mcnt    = 0;
                while (defined $str and length($str) + length($message) < 4096) {
                    $mcnt++ if $str;
                    $message .= $str;
                    $str = shift @$mlist;
                }
                $message =~ s/<TASKS>/$mcnt/;
                push @to_send, $message;
            }
            for my $message (@to_send) {
                $message =~ s/<PAGES>/$page/;
                send_message_to_telegram($chanel, $token, $message);
            }
        }
    }
}

sub get_filtred_startrek_tickets {
    my ($queue, $status) = @_;

    my @tickets;

    my $page  = 1;
    my $st    = get_yandex_startrek();
    my $query = sprintf '%s: notEmpty() and (%s) and (%s) and type: 1 "Sort By": Created',
      $FIELD_FOR_CHECK,
      join(' or ',  map {"Queue: $_"} @$queue),
      join(' and ', map {"Status: ! $_"} @$status),
      ;
    LOG $query;
    while (1) {
        my $tickets = $st->search(
            query    => $query,
            page     => $page,
            per_page => 50,
        );
        last unless @$tickets;
        push @tickets, @$tickets;
        LOG("page: $page count: " . scalar @$tickets);
        $page++;
    }

    return \@tickets;
}

sub get_yandex_startrek {

    my $ys = Yandex::StarTrek->new(
        oauth_token => get_secret('startrek-token'),
        retry       => 3,
        delay       => 3,
    );

    return $ys;
}

sub get_maked_data {

    my $path = lib::abs::path('data.json');
    my $data = read_file($path, {binmode => ':utf8', err_mode => 'quiet'});
    if ($data) {
        $data = decode_json($data);
    } else {
        $data = {};
    }

    return $data;
}

sub set_maked_data {
    my ($data) = @_;

    my $path = lib::abs::path('data.json');
    LOG($path);
    $data = JSON::PP->new->pretty->canonical->encode($data);
    write_file($path, {binmode => ':utf8'}, $data);
}

sub get_time {
    my ($str) = @_;

    my @tm = $str =~ /^(\d+)-(\d+)-(\d+)T(\d+):(\d+):(\d+)/;

    return timegm($tm[5], $tm[4], $tm[3], $tm[2], $tm[1] - 1, $tm[0] - 1900);
}

sub _logf {
    my ($severity, $pattern, @params) = @_;

    _log($severity, sprintf($pattern, @params), 1);
}

sub _log {
    my ($severity, $message, $logf) = @_;

    my @now = localtime;
    my $date = sprintf "%04d-%02d-%02d %02d:%02d:%02d", 1900 + $now[5], $now[4] + 1, $now[3], $now[2], $now[1], $now[0];

    if ($severity eq 'LOG') {
        $| = 1;
        my $str = sprintf("[%s]\t%s\t%s\n", $date, $$, $message);
        print $str;
    } else {
        my ($package, $filename, $line) = caller(1 + ($logf ? 1 : 0));
        my $str = sprintf("[%s]\t%s\t[%s:%s]\t%s\t%s\n", $date, $$, $package, $line, $severity, $message);
        warn $str;
    }
}

sub get_secret {
    my ($name) = @_;

    unless ($PI_SECRETS) {
        $PI_SECRETS = decode_json scalar read_file '/etc/pi-secrets.json';
    }

    if (exists $PI_SECRETS->{$name}) {
        return $PI_SECRETS->{$name};
    } else {
        $name //= 'undef';

        die "Can't find secret '$name'";
    }
}

sub send_message_to_telegram {
    my ($chanel, $token, $message) = @_;

    unless ($BROWSER) {
        $BROWSER = LWP::UserAgent->new(
            ssl_opts => {
                verify_hostname => 0,
                SSL_verify_mode => SSL_VERIFY_NONE
            }
        );
    }

    my $request = POST(
        "https://api.telegram.org/bot$token/sendMessage",
        [
            text                     => $message,
            chat_id                  => $chanel_target->{$chanel}{id},
            parse_mode               => 'HTML',
            disable_web_page_preview => 1,
        ]
    );

    unless ($REAL_SEND) {
        warn "to chanel: $chanel\n$message\n";
        return;
    }

    my $result = eval {$BROWSER->request($request)};
    WARN $@ if $@;

    if ($result and $result->code != 200) {
        ERROR "\n" . $message;
        ERROR Dumper($result);
    }

    return 1;
}

sub get_args {

    my $help = 0;
    GetOptions(
        'tg_chanel:s' => \$DEBUG_TG_CHANNEL,
        'send!'       => \$REAL_SEND,
        'help|?|h'    => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

}

__END__
