#!/usr/bin/perl
use strict;
use warnings FATAL => 'all';

use utf8;
use open qw(:std :utf8);

use JSON::XS;
use File::Slurp;

use Sys::RunAlone silent => 1;

my $secret;

main();

sub main {
    get_today_image_new();
}

sub exit_with_error {
    my ($message) = @_;

    warn $message . "\n";
    exit 1;
}

sub get_auth_data {
    my ($registry) = @_;
    my $secret = decode_json(read_file("/etc/registry_auth.json"));
    my ($username, $password, $email) = @{$secret->{$registry}}{qw(username password email)};
    return ($username, $password, $email);
}

sub get_token {
    return decode_json(read_file("/etc/pi-secrets.json"))->{'registry-token'};
}

sub today {
    my @tm = localtime;
    return sprintf "%04d-%02d-%02d", $tm[5]+1900, $tm[4]+1, $tm[3];
}

sub find_today_docker_image {
    my ($registry, $username, $password) = @_;

    my $dt = today();
    my $cmd = sprintf
        "curl -s --insecure --user %s:%s 'https://%s/v1/search?q=partner2-db'"
        . " | jq '.results[].name'"
        . " | grep '%s'"
        . " | sort",
        $username, $password, $registry, $dt;

    my @images = grep { $_ !~ /-null-/ } split /\n/, `$cmd`;

    my $last_image = pop @images;

    if ($last_image) {
        $last_image =~ s|^"library/||;
        $last_image =~ s|"$||;
    }
    else {
        warn "Can't find docker image for the date=$dt";
    }

    return $last_image;
}

sub find_today_docker_image_tags {
    my ($registry, $image, $token) = @_;

    my $dt = today();
    my $cmd = sprintf
        "curl -s -H 'Authorization: OAuth %s' 'https://%s/v2/%s/tags/list'"
        . " | jq '.tags'"
        . " | grep '%s'"
        . " | sort",
        $token, $registry, $image, $dt;

    my @images = split /\n/, `$cmd`;

    my $last_image = pop @images;

    if ($last_image) {
        $last_image =~ s|[\s",]||g;
    }
    else {
        warn "Can't find docker image for the date=$dt";
    }

    return "$image:$last_image";
}

sub pull_image {
    my ($image, $registry, $username, $password, $email) = @_;

    if ($image){
        my $exit_status = system(
            "docker login --username=$username --password=$password $registry &> /dev/null"
        );
        exit_with_error "Can't login to $registry" if $exit_status != 0;

        system("docker pull $image &> /dev/null");
    }
    return 1;
}

sub get_today_image_new {
    my $token = get_token();
    my $registry = 'registry.yandex.net';
    my $imagename = 'partners/partner2-db-general';
    my ($username, $password, $email) = get_auth_data($registry);
    my $image = find_today_docker_image_tags($registry, $imagename, $token);
    pull_image("$registry/$image", $registry, $username, $password, $email);
}

1;

__END__
