#!/usr/bin/perl

=head1 DESCRIPTION

    The script syncs a db from prod to ts, from ts to dev and creates docker images

=head1 USAGE

    /usr/bin/mysql-sync-data.pl

=head1 OPTIONS

    debug - debug mode

=cut


use strict;
use warnings FATAL => 'all';
use utf8;

use Data::Dumper;
use DBI;
use File::Path qw(make_path remove_tree);
use Getopt::Long qw();
use JSON::PP qw(encode_json decode_json);
use LWP::UserAgent;
use Pod::Usage;

use lib qw(/usr/lib/perl5/);

use PiSecrets;

my $DEBUG = 0;

my $META = {
    'test' => {
        transfer_ids => ['dttm234c79fhbl0sdq7k'],
        host        => '127.0.0.1',
        secret_name => 'connection-to-partner2-testing-database',
        prepare_db  => [
            \&prepare_partner_db_misc_sql,
            \&prepare_partner_db_grant_developer_roles,
            \&prepare_partner_db_docker_image,
        ],
        port        => 3478,
    },
    'dev'  => {
        transfer_ids => ['dttdpe8u5mh806c0rs3i',],
        host        => '127.0.0.1',
        secret_name => 'connection-to-partner2-development-database',
        prepare_db  => [
            \&prepare_partner_db_misc_sql,
            \&prepare_partner_db_grant_developer_roles,
        ],
        port        => 3678,
    },
};

my $MYSQL_PORT = 3308;
my $MYSQL_USER = 'root';

my $DUMP_DIR = '/opt/partner2-beta-docker/general_beta_dump';

my $SOURCE_DIR = '/opt/prepare-db/docker_db';

my $REPOSITORY = "registry.yandex.net/partners";

my $IMAGE_PREFIX = "partner2-db";

my $GENERAL_IMAGE_DIR = '/opt/partner2-beta-docker/general_beta_image';
my $GENERAL_IMAGE_NAME = "$IMAGE_PREFIX-general";
my $GENERAL_DATA_DIR = "$GENERAL_IMAGE_DIR/mysql.partner2-beta-base";

my $MENTOL_IMAGE_DIR = '/opt/partner2-beta-docker/mentol_beta_image';
my $MENTOL_IMAGE_NAME = "$IMAGE_PREFIX-mentol";
my $MENTOL_DATA_DIR = "$MENTOL_IMAGE_DIR/mysql.partner2-beta-base";

my $AUTOTEST_IMAGE_NAME = "$IMAGE_PREFIX-autotest";

my $MYSQL_UID = 101;
my $MYSQL_GID = 102;

my $HOST = 'https://cdc.n.yandex-team.ru';

my $STATUS_SNAPSHOTTING = 'SNAPSHOTTING';
my $STATUS_OK = 'DONE';
my $STATUS_ERROR = 'ERROR';

my @TECH_LOGINS = qw(system-cron yndx-partner-intapi yndx-robot-adfox-pibot);
my $TECH_LOGINS = join(', ', map {"'$_'"} @TECH_LOGINS);
my $TECH_NOTIFICATION_TYPE = 'auto';

my $FILES_FILTER = q(*.sql);

my $INSTANCE = 'general-beta-base';
my $MYSQL_DIR = "/tmp/partner2-beta-docker/$INSTANCE";
my $USER = 'mysql';
my $CONFIG = "$MYSQL_DIR/$INSTANCE.cnf";
my $SOCKET = "$MYSQL_DIR/mysqld-$INSTANCE.sock";
my $TEMPLATE = '/opt/prepare-db/partnerdb2.cnf.template';

my $PROD_BUILD_VERSION_URL = 'https://partner2.yandex.ru/intapi/devel/version.json';

my @EMPTY_TABLES = qw(
    email
    email_attachment
    email_data
    event_log
    pagesbs
    quality_pages_coef_arc
    queue
    ssp_impression_log
    stat_report_params_digest
);

my @NOT_HEAVY_STAT_TABLES = qw(statistics_reports statistics_reports_level_order);

#### Данные справочных таблиц
my @DICTIONARY_TABLES = qw(
    bk_language
    currencies
    dsp
    dsp_type
    focus_domains_for_managers_mailing_lists
    geo_base
    intapi_acl
    kv_store
    migrations
    monetizers
    picategories_dict
    statistics_reports_level_order
    text_template
    thematic_filters
    tns_dict_article
    tns_dict_brand
    ya_categories
);

my %LIMIT_100_TABLES = map {$_ => 1} qw(
    block_tags
    dictionary_words
    internal_content
    internal_content_genres
    partner_and_internal_genre
    partner_content
    partner_genre
    ssp_moderation
    video_stat_files
);

main();

sub main {
    my $args = _get_args();
    $DEBUG = $args->{'debug'};

    p('START');

    eval {

        sync_data('test');
        sync_data('dev');

        send_juggler_event('OK', 'Successful data sync');
    };

    if ($@) {
        send_juggler_event('CRIT', $@);
        die $@;
    }

    p('END');
}

sub _get_args {
    my $args = {};

    my $result = Getopt::Long::GetOptions(
        'debug!'   => \$args->{'debug'},
        'help|?|h' => \$args->{'help'},
    );

    if (!$result || $args->{'help'}) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }

    return $args;
}

sub sync_data {
    my ($stage) = @_;

    p("start sync data for '$stage'");

    foreach my $transfer_id (@{$META->{$stage}{'transfer_ids'}}) {
        unless(defined($transfer_id)) {
            p('transfer_id not found', {die => 1});
        }

        if ($DEBUG) {
            p('DEBUG MODE: transfer is not going to start');
            next;
        }

        truncate_table_pages($stage);

        my $operation_id = activate_transfer($transfer_id);
        p("Transfer '$transfer_id' activated, operation id is '$operation_id'");

        #TODO: спросить как узнать когда отработала
        wait_success_transfer($transfer_id);
    }

    prepare_db($stage);

    p("end sync data for '$stage'");
}

sub truncate_table_pages {
    my ($stage) = @_;

    my $dbh = get_connection_to_mysql($stage, 'get_dbh');

    my $sth = $dbh->prepare("show tables like 'pages';") or die $dbh->errstr();

    $sth->execute() or die $sth->errstr();

    my $data = $sth->fetchall_arrayref({}) or die $sth->errstr();

    if (@$data) {
        p("truncate table 'pages'");
        $dbh->do("truncate pages;") or die $dbh->errstr();
    }
}

sub prepare_db {
    my ($stage) = @_;

    my $prepare_db = $META->{$stage}{'prepare_db'};

    foreach my $sub (@$prepare_db) {
        $sub->($stage);
    }
}

sub ts {
    my ($stage, $all_sqls) = @_;

    my @sqls = sort grep { $_ =~ /\/PI-/ } @$all_sqls;
    push (@sqls, sort grep { $_ =~ /\/INFRASTRUCTUREPI-/ } @$all_sqls);

    my $connection = get_connection_to_mysql($stage);

    map {
        my $sql = $_;
        p("Executing $sql...");

        run_shell("cat $sql | $connection 2>&1");
    } @sqls;
}

sub dev {
    my ($stage, $all_sqls) = @_;

    my @sqls = sort grep { $_ =~ /\/PI-/ && $_ !~ /_MANUAL/ } @$all_sqls;
    push (@sqls, sort grep { $_ =~ /\/INFRASTRUCTUREPI-/ && $_ !~ /_MANUAL/ } @$all_sqls);

    my $connection = get_connection_to_mysql($stage);

    map {
        my $sql = $_;
        p("Executing $sql...");

        run_shell("cat $sql | $connection 2>&1");
    } @sqls;
}

sub prepare_partner_db_docker_image {
    my ($stage) = @_;

    partner_db_dump($stage);

    make_partner_db_beta($stage);

    make_general_image();
    make_mentol_image();
    make_autotest_image();
}

sub get_db_tables {
    my ($stage, $pattern, $table_names) = @_;

    my $query = qq{
SELECT  table_name
FROM    information_schema.tables
WHERE   TABLE_SCHEMA = 'partner'
    };

    if (defined($pattern)) {
        $query .= " AND table_name LIKE '$pattern'";
    }

    if (defined($table_names)) {
        $query .= " AND table_name NOT IN ( " . join(', ', map {"'$_'"} @$table_names) . " )";
    }

    my $dbh = get_connection_to_mysql($stage, 'get_dbh');

    my $sth = $dbh->prepare($query) or die $dbh->errstr();

    $sth->execute() or die $sth->errstr();

    my $data = $sth->fetchall_arrayref({}) or die $sth->errstr();

    return [map {$_->{'table_name'}} @$data];
}

# Определение тега релиза в проде
sub get_prod_build_version {
    my $response = run_shell(qq{curl -s "$PROD_BUILD_VERSION_URL"});
    my $data = eval { decode_json($response) };
    my $release_tag;
    $release_tag = $data->{data} if (ref($data) eq 'HASH' && ($data->{result} // '') eq 'ok');
    if ($release_tag) {
        p("Release tag: $release_tag ($PROD_BUILD_VERSION_URL)");
    } else {
        p('Can not get release tag', {die => 1}, $response);
    }

    return $release_tag;
}

sub partner_db_dump {
    my ($stage) = @_;

    #/opt/prepare-db/docker_db/general/partnerdb-dump.sh

    make_path($DUMP_DIR) unless -d $DUMP_DIR;

    foreach (qw(kladr partner_logs partner)) {
        remove_tree("$DUMP_DIR/$_");

        make_path("$DUMP_DIR/$_/triggers");
    }

    ######################################################
    #### Только структура (пустые таблицы) ###############
    ######################################################

    #### Дампим схемы баз

    foreach my $database (qw(partner_logs kladr partner)) {
        p("Start dumping structure '$database' DB tables");

        _dump($stage, {
            database      => $database,
            no_data       => 1,
            skip_triggers => 1,
            file          => "$DUMP_DIR/$database/all_struct.sql"
        });
    }

    ######################################################
    #### Таблицы с данными ###############################
    ######################################################

    my @empty_tables = (@EMPTY_TABLES, @{get_db_tables($stage, '%_action_log%')});

    p('Empty tables', {}, @empty_tables);

    my $heavy_tables = get_db_tables($stage, 'statistics_%', [@empty_tables, @NOT_HEAVY_STAT_TABLES]);

    p('Start dumping "partner" DB dictionary tables:');
    my $counter = 0;
    foreach my $table_name (@DICTIONARY_TABLES) {
        p(++$counter . ". $table_name");

        my $where = '1';

        if ($DEBUG) {
            $where = "$where limit 10";
        }

        _dump($stage, {
            database       => 'partner',
            no_create_info => 1,
            skip_triggers  => 1,
            table          => $table_name,
            where          => $where,
            file           => "$DUMP_DIR/partner/dictionary_tables.sql"
        });
    }

    my $user_where = "(login in ($TECH_LOGINS) OR id in (SELECT owner_id FROM partner.dsp))";
    my $user_where_not = "(login not in ($TECH_LOGINS) AND id not in (SELECT owner_id FROM partner.dsp))";

    my %tables = (
        users        => {
            name                   => 'users',
            condition => $user_where,
            negative_condition => $user_where_not,
        },
        user_role    => {
            name                   => 'user_role',
            condition => "user_id in (SELECT id FROM partner.users WHERE $user_where)",
            negative_condition => "user_id not in (SELECT id FROM partner.users WHERE $user_where)",
        },
        notification => {
            name                   => 'notification',
            condition => "type in ('$TECH_NOTIFICATION_TYPE')",
            negative_condition => "type not in ('$TECH_NOTIFICATION_TYPE')",
        },
    );

    foreach (values(%tables)) {
        my $table_name = $_->{'name'};

        p(++$counter . ". $table_name");

        _dump($stage, {
            database       => 'partner',
            no_create_info => 1,
            skip_triggers  => 1,
            table          => $table_name,
            where          => $_->{'condition'},
            file           => "$DUMP_DIR/partner/dictionary_tables.sql"
        });
    }

    #### Данные основных таблиц
    p('Start dumping "partner" DB basic tables:');

    my $basic_tables = get_db_tables($stage, undef, [@empty_tables, @$heavy_tables, @DICTIONARY_TABLES]);

    $counter = 0;
    foreach my $table_name (@$basic_tables) {
        p(++$counter . ". $table_name");

        my $where = '1';

        if (exists($tables{$table_name})) {
            $where = "$where AND $tables{$table_name}->{'negative_condition'}";
        }

        if (exists($LIMIT_100_TABLES{$table_name}) || $DEBUG) {
            $where = "$where limit 100";
        }

        _dump($stage, {
            database       => 'partner',
            no_create_info => 1,
            skip_triggers  => 1,
            table          => $table_name,
            where          => $where,
            file           => "$DUMP_DIR/partner/basic_tables.sql"
        });
    }

    ### Триггеры
    _dump_triggers($stage);

    # Определение тега релиза в проде
    my $release_tag = get_prod_build_version();

    run_shell(qq{echo "$release_tag" > $DUMP_DIR/release.tag});
}

sub _dump_triggers {
    my ($stage) = @_;

    _dump($stage, {
        database       => 'partner',
        triggers       => 1,
        no_create_info => 1,
        no_data        => 1,
        file           => "$DUMP_DIR/partner/triggers/triggers.sql"
    });

    my $secrets = get_secret($META->{$stage}{'secret_name'});

    run_shell(qq{sed -i 's/$secrets->{user}/$MYSQL_USER/'g $DUMP_DIR/partner/triggers/triggers.sql});
}

sub _dump {
    my ($stage, $params) = @_;

    my ($database, $table, $where, $file,
        $no_data, $no_create_info, $triggers, $skip_triggers) = @$params{qw(database table where file no_data no_create_info triggers skip_triggers)};

    my $host = $META->{$stage}{'host'};

    my $secrets = get_secret($META->{$stage}{'secret_name'});

    my $mysql_port = $META->{$stage}{'port'};

    my $cmd = "MYSQL_PWD=$secrets->{'password'} mysqldump --host=$host --port=$mysql_port --user=$secrets->{'user'}"
        . ($no_data ? " --no-data" : '')
        . ($no_create_info ? " --no-create-info" : '')
        . ($triggers ? " --triggers" : '')
        . ($skip_triggers ? " --skip_triggers" : '')
        . " --single-transaction --quick --set-gtid-purged=OFF $database"
        . (defined($table) ? " \"$table\"" : '')
        . (defined($where) ? " --where=\"$where\"" : '')
        . " >> $file || true";

    run_shell($cmd);
}

sub make_partner_db_beta {
    my ($stage) = @_;

    #/opt/prepare-db/docker_db/general/make-partnerdb-beta.sh

    ##############################
    #####   GENERAL   ############
    ##############################

    p('Start created GENERAL beta mysql db');

    cleanup_db($GENERAL_DATA_DIR);
    init_db_dirs($GENERAL_DATA_DIR);
    prepare_db_config($GENERAL_DATA_DIR);

    p('starting db');
    start_db($GENERAL_DATA_DIR);

    foreach my $db (qw(kladr partner partner_logs)) {
        p("start create db $db");

        run_shell(qq{echo "create database $db" | mysql -u $MYSQL_USER -S $SOCKET});

        unless(fork()) {
            p("start upload db $db");

            run_shell(qq{nohup cat $DUMP_DIR/$db/*.sql | mysql -u $MYSQL_USER -S $SOCKET --force $db 2>&1});

            exit;
        }
    }

    while (1) {
        my $result = run_shell(qq{ps -e -o pid,cmd | grep "mysq[l] -u $MYSQL_USER -S $SOCKET"}, skip_exit_code => 1);

        last if $result eq '';

        sleep(15);
    }

    run_shell(qq{nohup cat $DUMP_DIR/partner/triggers/triggers.sql | mysql -u $MYSQL_USER -S $SOCKET --force partner 2>&1});

    p('stop mysql');
    stop_db();

    p("mysql beta created in $GENERAL_DATA_DIR");

    ##############################
    #####    MENTOL   ############
    ##############################

    p('Start created MENTOL beta mysql db');

    cleanup_db($MENTOL_DATA_DIR);
    init_db_dirs($MENTOL_DATA_DIR);
    prepare_db_config($MENTOL_DATA_DIR);

    p('starting db');
    start_db($MENTOL_DATA_DIR);

    foreach my $db (qw(kladr partner partner_logs)) {
        p("start create db $db");

        run_shell(qq{echo "create database $db" | mysql -u $MYSQL_USER -S $SOCKET});


        unless(fork()) {
            p("start upload db $db");

            run_shell(qq{nohup cat $DUMP_DIR/$db/all_struct.sql | mysql -u $MYSQL_USER -S $SOCKET --force $db 2>&1});

            exit;
        }
    }

    while (1) {
        my $result = run_shell(qq{ps -e -o pid,cmd | grep "mysq[l] -u $MYSQL_USER -S $SOCKET"}, skip_exit_code => 1);

        last if $result eq '';

        sleep(15);
    }

    run_shell(qq{nohup cat $DUMP_DIR/partner/dictionary_tables.sql | mysql -u $MYSQL_USER -S $SOCKET --force partner 2>&1});

    run_shell(qq{nohup cat $DUMP_DIR/partner/triggers/triggers.sql | mysql -u $MYSQL_USER -S $SOCKET --force partner 2>&1});

    p('stop mysql');
    stop_db();

    p("mysql beta created in $MENTOL_DATA_DIR");
}

sub make_general_image {

    #/opt/prepare-db/docker_db/general/build_partner_db_docker_images.sh

    # Delete old images
    p("Start delete old images");
    run_shell("docker images | grep $IMAGE_PREFIX | grep 'months ago' | awk '{print \$3}' | xargs -n 1 docker rmi", skip_exit_code => 1);


    # Собираем докерный образ БД
    my $date = get_date(without_time => 1);

    my $release_tag = run_shell("cat $DUMP_DIR/release.tag");

    my $image_tag = "$release_tag-$date";

    my $GENERAL_IMAGE_NAME_WITH_TAG = "${GENERAL_IMAGE_NAME}-${image_tag}";

    run_shell("cp $SOURCE_DIR/Dockerfile.partner-docker-db  $GENERAL_IMAGE_DIR/Dockerfile");
    run_shell("cp $SOURCE_DIR/partner-docker-my.cnf         $GENERAL_IMAGE_DIR/my.cnf");

    p('Start create GENERAL archve with db files');

    run_shell("rm -f $GENERAL_IMAGE_DIR/docker_database.tar");
    chdir("$GENERAL_IMAGE_DIR/mysql.partner2-beta-base/") or die "Failed to chdir to $GENERAL_IMAGE_DIR/mysql.partner2-beta-base/: $!";
    my @tar_content = glob("*");
    run_shell("tar cf $GENERAL_IMAGE_DIR/docker_database.tar " . join(' ', @tar_content) . " --hard-dereference --owner=$MYSQL_UID --group=$MYSQL_GID");
    chdir($ENV{'HOME'});

    p('Start build GENERAL docker image');
    run_shell("docker build -q --tag registry.yandex.net/partners/$GENERAL_IMAGE_NAME:$image_tag $GENERAL_IMAGE_DIR/");

    p("Start push docker image ($date, $release_tag)");
    run_shell("docker push registry.yandex.net/partners/$GENERAL_IMAGE_NAME:$image_tag", dry_run => $DEBUG ? 1 : undef);

    p("GENERAL docker image created - $GENERAL_IMAGE_NAME:$image_tag");
}

sub make_mentol_image {

    my $date = get_date(without_time => 1);
    my $release_tag = run_shell("cat $DUMP_DIR/release.tag");
    my $image_tag = "$release_tag-$date";

    run_shell("cp $SOURCE_DIR/Dockerfile.partner-docker-db  $MENTOL_IMAGE_DIR/Dockerfile");
    run_shell("cp $SOURCE_DIR/partner-docker-my.cnf         $MENTOL_IMAGE_DIR/my.cnf");

    p('Start create MENTOL archve with db files');

    run_shell("rm -f $MENTOL_IMAGE_DIR/docker_database.tar");
    chdir("$MENTOL_IMAGE_DIR/mysql.partner2-beta-base/") or die "Failed to chdir to $MENTOL_IMAGE_DIR/mysql.partner2-beta-base/: $!";
    my @tar_content = glob("*");
    run_shell("tar cf $MENTOL_IMAGE_DIR/docker_database.tar " . join(' ', @tar_content) . " --hard-dereference --owner=$MYSQL_UID --group=$MYSQL_GID");
    chdir($ENV{'HOME'});

    p('Start build MENTOL docker image');
    run_shell("docker build -q --tag registry.yandex.net/partners/$MENTOL_IMAGE_NAME:$image_tag $MENTOL_IMAGE_DIR/");

    p("Start push MENTOL docker image to registry.yandex.net ($date, $release_tag)");
    run_shell("docker push registry.yandex.net/partners/$MENTOL_IMAGE_NAME:$image_tag", dry_run => $DEBUG ? 1 : undef);

    p('End');
}

sub make_autotest_image {

    my $date = get_date(without_time => 1);

    my $release_tag = run_shell("cat $DUMP_DIR/release.tag");

    my $image_tag = "$release_tag-$date";

    my $full_mentol_image_name = "$REPOSITORY/$MENTOL_IMAGE_NAME";

    my $full_autotest_image_name = "$REPOSITORY/$AUTOTEST_IMAGE_NAME:$image_tag";

    my $socket_dir = "/tmp/$$/var/run";
    my $socket_name = "mysqld.sock";
    my $socket_mysql = "$socket_dir/$socket_name";

    my $mysql_connect = "mysql -u $MYSQL_USER -S $socket_mysql partner";

    # Create autotest image
    p("Start create autotest image");
    make_path($socket_dir);
    run_shell("chmod a+rwx $socket_dir");

    p('RUN mentol image');
    run_shell("docker rm -f $full_autotest_image_name", skip_exit_code => 1);
    run_shell("docker rm -f $AUTOTEST_IMAGE_NAME", skip_exit_code => 1);

    run_shell("docker run --detach -v $socket_dir:/var/run/mysqld --name $AUTOTEST_IMAGE_NAME.$date $full_mentol_image_name:$image_tag");

    p("wait to connect '$mysql_connect'");
    my $res = "";
    my $time = time + 300;
    while ($res ne "1" && time < $time) {
        if (-S $socket_mysql) {
            $res = run_shell("echo 'select 1' | $mysql_connect | tail -n +2");
        }

        sleep(5);
    }

    die 'MySQL does not start into docker' if $res ne "1";

    p('make changes');

    my $sql_dir = "/opt/prepare-db";
    # для тестирование использовать этот путь
    # sql_dir="partner-db"

    run_shell("cat $sql_dir/partner_autotest.sql | $mysql_connect");
    my $res_tables = run_shell("cat $sql_dir/partner_autotest_truncate.sql | $mysql_connect | tail -n +2");
    foreach my $table (grep {defined($_)} split(/\s+/, $res_tables)) {
        my $sql = "SET FOREIGN_KEY_CHECKS = 0;truncate table \`partner\`.\`$table\`;SET FOREIGN_KEY_CHECKS = 1;";

        run_shell("echo '$sql' | $mysql_connect");
    }
    # Добавляем в kv_store время создания базы
    run_shell("cat $sql_dir/partner_autotest_db_inittime.sql | $mysql_connect");


    p('commit image');
    run_shell("docker stop $AUTOTEST_IMAGE_NAME.$date");
    run_shell("docker commit $AUTOTEST_IMAGE_NAME.$date $full_autotest_image_name");
    run_shell("docker rm $AUTOTEST_IMAGE_NAME.$date");

    p("Start push AUTOTEST docker image to registry.yandex.net ($date, $release_tag)");
    run_shell("docker push $full_autotest_image_name", dry_run => $DEBUG ? 1 : undef);

    remove_tree($socket_dir);

    # Delete old images
    p('Start delete old images');
    run_shell("docker images | grep $AUTOTEST_IMAGE_NAME | grep 'months ago' | awk '{print \$3}' | xargs -n 1 docker rmi", skip_exit_code => 1);

    p('End');
}

sub prepare_partner_db_misc_sql {
    my ($stage) = @_;

    p("start prepare_partner_db_misc_sql for '$stage'");

    my $connection = get_connection_to_mysql($stage);

    run_shell("$connection < /opt/prepare-db/grant-developer-roles.sql");

    # https://st.yandex-team.ru/TM-1174
    my $sql = 'update users set id = 0 where login = "system-cron"';
    run_shell("echo '$sql' | $connection");
}

sub prepare_partner_db_grant_developer_roles {
    my ($stage) = @_;

    p("start prepare_partner_db_grant_developer_roles for '$stage'");

    my $connection = get_connection_to_mysql($stage);

    run_shell("$connection < /opt/prepare-db/partner_misc.sql");
}

sub run_shell {
    my ($shell_command, %opts) = @_;

    my $silent  = $opts{'silent'};
    my $dry_run = $opts{'dry_run'};

    local $| = 1;
    local $/ = " ";

    print "$shell_command\n" unless $silent;

    return $dry_run if defined($dry_run);

    open(my $fh, '-|', $shell_command) or die "OPEN: $!";

    my $result = '';
    while (my $line = <$fh>) {
        $result .= $line;
        print $line unless $silent;
    }
    print "\n" unless $silent || $result =~ /\n\r?$/;

    unless (close($fh)) {
        if ($!) {
            die sprintf("Shell command `%s` error close: %s", $shell_command, $!);
        } elsif (!$opts{skip_exit_code}) {
            die sprintf("Shell command `%s` returned a non zero exit status: %d", $shell_command, $? >> 8);
        }
    }

    $result =~ s/\s*$//;

    return $result;
}

sub get_connection_to_mysql {
    my ($stage, $get_dbh) = @_;

    my $host = $META->{$stage}{'host'};

    my $secret_name = $META->{$stage}{'secret_name'};

    my $secrets = get_secret($secret_name);

    my $mysql_port = $META->{$stage}{'port'};

    if ($get_dbh) {
        my $dsn = "DBI:mysql:database=partner;host=$host;port=$mysql_port";

        my $dbh = DBI->connect(
            $dsn,
            $secrets->{'user'},
            $secrets->{'password'},
            {
                PrintError => 0,
                RaiseError => 0,
                AutoCommit => 1,
            },
        ) or die DBI::err();

        $dbh->{'mysql_auto_reconnect'} = '';
        $dbh->do('SET NAMES utf8');
        $dbh->{'mysql_enable_utf8'} = 1;

        return $dbh;
    } else {
        return "MYSQL_PWD=$secrets->{'password'} mysql -fB --host=$host --user=$secrets->{'user'} --port=$mysql_port partner";
    }
}

sub activate_transfer {
    my ($transfer_id) = @_;

    p("activate transfer '$transfer_id'");

    my $response = request({
        method => 'PATCH',
        headers => {Authorization => 'OAuth ' . get_secret('transfer-manager-token')},
        url => "$HOST/v1/transfer/activate",
        data => encode_json({transfer_id => $transfer_id})
    });

    return decode_json($response)->{'id'};
}

sub wait_success_transfer {
    my ($transfer_id) = @_;

    p("Wait transfer '$transfer_id'");

    my $status;

    my $time = time() + 6 * 60 * 60;

    while (time() < $time) {
        my $json_response = request({
            method  => 'GET',
            headers => { Authorization => 'OAuth ' . get_secret('transfer-manager-token') },
            url     => "$HOST/v1/transfer/$transfer_id",
        });

        my $response = decode_json($json_response);
        $status = $response->{'status'};

        if ($status eq $STATUS_OK) {
            p("Transfer '$transfer_id' is done");

            last;
        } elsif ($status eq $STATUS_ERROR) {
            p("Transfer '$transfer_id' is done with error", {die => 1});
            p($json_response);
        }

        sleep(300);
    }

    unless ($status) {
        p("Transfer '$transfer_id' is not done", {die => 1});
    }
}

sub request {
    my ($r) = @_;

    my $http_headers = HTTP::Headers->new(
        Accept        => 'application/json',
        Content_Type  => 'application/json',
        %{$r->{'headers'} // {}},
    );

    my $http_request = HTTP::Request->new($r->{'method'} => $r->{'url'}, $http_headers, $r->{'data'});

    my $lwp = LWP::UserAgent->new();

    #utf8::encode($sql);

    my $retries = 0;

    my $content;
    while (++$retries < 4) {
        my $response = $lwp->request($http_request);

        $content = $response->decoded_content;

        if ($response->is_success) {
            last;
        }

        p("Got answer: $content");

        $content = undef;

        sleep($retries * 3);
    }

    unless (defined($content)) {
        p('Can not get response', {die => 1});
    }

    return $content;
}

sub p {
    my ($msg, $options, @data) = @_;

    if (ref($msg)) {
        $msg = Dumper($msg);
    }

    if (@data) {
        $msg .= ' ' . Dumper(\@data);
    }

    $msg = get_date() . " - $msg\n";

    if ($options->{'die'}) {
        die $msg;
    } else {
        print $msg;
    }
}

sub get_date {
    my (%opts) = @_;

    my @localtime = localtime();

    @localtime = reverse splice(@localtime, 0, 6);

    $localtime[0] += 1900;
    $localtime[1] += 1;

    my $date = sprintf('%4d-%02d-%02d %02d:%02d:%02d', @localtime);

    if ($opts{'without_time'}) {
        ($date) = $date =~ /^(\d+-\d+-\d+)\s/;
    }

    return $date;
}

sub prepare_db_config {
    my ($datadir) = @_;

    #TODO: прочитать, проапдейтить и сохранить (либо Template) + что-то сразу в конфиг оно не меняется...

    run_shell("cp $TEMPLATE $CONFIG");

    run_shell(qq{perl -i -pe "s|LIBDIR|$MYSQL_DIR|g" $CONFIG});
    run_shell(qq{perl -i -pe "s|INSTANCE|$INSTANCE|g" $CONFIG});

    run_shell(qq{perl -i -pe "s|DATADIR|$datadir|g" $CONFIG});
    run_shell(qq{perl -i -pe "s|USER|$USER|g" $CONFIG});

    #replace 3306 to MYSQL_PORT
    run_shell(qq{perl -i -pe "s|3306|$MYSQL_PORT|g" $CONFIG});
}

sub start_db {
    my($datadir) = @_;

    unless (-d "$datadir/mysql") {
        run_shell("/usr/sbin/mysqld --initialize-insecure --user=$USER --datadir=$datadir");
    }

    unless (-S $SOCKET) {
        unless (fork()) {
            run_shell("nohup /usr/sbin/mysqld --defaults-file=$CONFIG --datadir=$datadir");

            exit;
        }
    }

    my $response = "0";
    my $time = time + 300;
    while ($response ne '1' && time < $time) {
        $response = run_shell(qq{echo "select 1" | mysql -S $SOCKET -u $MYSQL_USER --skip-column-names}, skip_exit_code => 1);

        sleep(5);
    }

    die 'Mysql does not start' if $response ne '1';
}

sub stop_db {
    if (-S $SOCKET) {
        run_shell("mysqladmin -u $MYSQL_USER -S $SOCKET shutdown");
    }
}

sub init_db_dirs {
    my ($datadir) = @_;

    make_path($MYSQL_DIR, $datadir);

    run_shell("touch $MYSQL_DIR/$INSTANCE.err");

    run_shell("chown mysql: -R $MYSQL_DIR");
    run_shell("chown mysql: -R $datadir");
}

sub cleanup_db {
    my ($datadir) = @_;

    stop_db();

    remove_tree($MYSQL_DIR, $datadir);
}

sub send_juggler_event {
    my ($status, $description) = @_;

    # escape single quote
    $description =~ s/'/'"'"'/g;

    run_shell(
        "/usr/bin/juggler_queue_event --host `hostname -f` --service mysql-sync-data --status $status --description '$description'",
        dry_run => $DEBUG ? 1 : undef
    );
}
