package infra;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.defaultconfiguration.PartnerLocalDateTime;

@Service
public class InfraApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(InfraApi.class);
    private final WebClient webClient;

    public InfraApi(@Value("{infra.api.url}") String baseUrl) {
        this.webClient = WebClient.create(baseUrl);
    }

    public List<InfraEvent> getEvents(int environmentId, LocalDateTime from, LocalDateTime to) {
        var responseEntity = webClient
                .get()
                .uri(uriBuilder -> uriBuilder.path("v1/events")
                        .queryParam("environmentId", environmentId)
                        .queryParam("from", from.toEpochSecond(ZoneOffset.UTC))
                        .queryParam("to", to.toEpochSecond(ZoneOffset.UTC))
                        .build()
                )
                .retrieve()
                .toEntityList(InfraEvent.class)
                .block();

        if (responseEntity == null) {
            throw new InfraException("Response is null");
        }

        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            throw new InfraException("Response status is " + responseEntity.getStatusCode().value());
        }

        return responseEntity.getBody();
    }

    public static void main(String[] args) {
        // TODO just code for future cron
        InfraApi infraApi = new InfraApi("https://infra-api-test.yandex-team.ru/");
        int dcOutageDrillsEnvId = 204;
        List<InfraEvent> infraEvents = infraApi.getEvents(
                dcOutageDrillsEnvId,
                PartnerLocalDateTime.now(),
                PartnerLocalDateTime.now().plusDays(2L)
        );

        for (InfraEvent infraEvent : infraEvents) {
            if (infraEvent.getStartTime().isBefore(PartnerLocalDateTime.now().plusDays(1L))) {
                LOGGER.error("CRIT! DС OUTAGE IS SOON");
            } else {
                LOGGER.warn("WARN! dс outage will be in a day");
            }
        }
    }
}
