package ru.yandex.partner.hourglass.configuration;

import java.time.Clock;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.List;

import javax.annotation.PostConstruct;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Profile;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.SchedulerInstancePinger;
import ru.yandex.direct.hourglass.SchedulerService;
import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.InstanceMetaImpl;
import ru.yandex.direct.hourglass.implementations.MD5Hash;
import ru.yandex.direct.hourglass.implementations.SchedulerInstancePingerImpl;
import ru.yandex.direct.hourglass.implementations.SchedulerServiceImpl;
import ru.yandex.direct.hourglass.implementations.TaskThreadPoolImpl;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.implementations.randomchoosers.RendezvousRandomChooser;
import ru.yandex.direct.hourglass.implementations.updateschedule.SimpleScheduleUpdateServiceImpl;
import ru.yandex.direct.hourglass.mysql.schedulerinstances.MysqlSchedulerInstancesRepository;
import ru.yandex.direct.hourglass.mysql.storage.StorageImpl;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;
import ru.yandex.direct.juggler.JugglerSender;
import ru.yandex.direct.scheduler.JobInterceptorsList;
import ru.yandex.direct.scheduler.SpringAppTerminator;
import ru.yandex.direct.scheduler.hourglass.ScheduleRecordListProvider;
import ru.yandex.direct.scheduler.hourglass.TaskFactory;
import ru.yandex.direct.scheduler.hourglass.TaskListProvider;
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;
import ru.yandex.direct.scheduler.hourglass.implementations.JobScheduleInfoFactory;
import ru.yandex.direct.scheduler.hourglass.implementations.ScheduleInfoProcessor;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskListProviderImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskParameterizer;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.MonitoringWriterImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.SchedulerPingListener;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.SchedulerPingMonitoring;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.CheckInSchdedulerPingListener;
import ru.yandex.direct.scheduler.hourglass.implementations.monitoring.juggler.HourglassJugglerSender;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.ScheduleInfoConverterImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.NextRunModifierFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.ByCronExpressionStrategy;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.NextRunStrategiesFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.PeriodicStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfoConverter;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifier;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifierFactory;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunCalcStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunStrategiesFactory;
import ru.yandex.direct.scheduler.support.JobExecutingMonitor;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.library.svnversion.VcsVersion;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.partner.hourglass.configuration.property.SchedulerProperty;

@Configuration
@Profile("!debugjob")
@ComponentScan(
        basePackages = "ru.yandex.direct.scheduler",
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
                @ComponentScan.Filter(classes = JobExecutingMonitor.class, type = FilterType.ASSIGNABLE_TYPE),
        }
)
@ConditionalOnProperty(
        value = "scheduler.enabled",
        havingValue = "true",
        matchIfMissing = true)
public class HourglassConfiguration {
    private static final Duration JOB_EXECUTION_MONITOR_INTERVAL = Duration.ofSeconds(10);
    private static final VcsVersion VCS = new VcsVersion(HourglassConfiguration.class);
    public static final String JOBS_VERSION = Integer.toString(VCS.getArcadiaLastChangeNum());

    private static final Logger LOGGER = LoggerFactory.getLogger(HourglassConfiguration.class);
    private final SpringAppTerminator springAppTerminator;
    @Autowired
    private HourglassScheduler hourglassScheduler;
    @Autowired
    private ThreadsHierarchy threadsHierarchy;

    public HourglassConfiguration(@Autowired SpringAppTerminator springAppTerminator) {
        this.springAppTerminator = springAppTerminator;
    }

    @Bean
    public InstanceId instanceId() {
        return new InstanceIdImpl();
    }

    @Bean
    public InstanceMeta instanceMeta() {
        var host = SystemUtils.strictHostname();
        var startTime = LocalDateTime.now(Clock.systemUTC()).toString();
        return new InstanceMetaImpl(host, startTime);
    }

    @Bean
    public HourglassProperties hourglassProperties(
            SchedulerProperty schedulerProperty
    ) {
        return HourglassProperties.builder()
                .setTaskFetchingInterval(schedulerProperty.getTaskFetchingIntervalMs(), ChronoUnit.MILLIS)
                .setReschedulingTasksInterval(schedulerProperty.getReschedulingTasksIntervalSec(), ChronoUnit.SECONDS)
                .setMaxHeartbeatAge(schedulerProperty.getMaxHeartbeatAgeSec(), ChronoUnit.SECONDS)
                .setPingInterval(schedulerProperty.getPingIntervalMs(), ChronoUnit.MILLIS)
                .setJobsFreeingInterval(schedulerProperty.getJobsFreeingIntervalSec(), ChronoUnit.SECONDS)
                .setInstancePingInterval(schedulerProperty.getInstancePingIntervalMs(), ChronoUnit.MILLIS)
                .setScheduleUpdateInterval(schedulerProperty.getScheduleUpdateIntervalMin(), ChronoUnit.MINUTES)
                .setScheduleHeartbeatExpiration(schedulerProperty.getScheduleHeartbeatExpirationMin(),
                        ChronoUnit.MINUTES)
                .setMissedTaskThreshold(schedulerProperty.getMissedTaskThresholdSec(), ChronoUnit.SECONDS)
                .setThreadPoolSize(schedulerProperty.getThreadPoolSize())
                .setLeaderVotingLowerBound(schedulerProperty.getLeaderVotingLowerBound())
                .build();
    }

    @Bean
    public ScheduleUpdateService scheduleUpdateService(Storage storage) {
        return new SimpleScheduleUpdateServiceImpl(storage);
    }

    @Bean
    public HourglassScheduler hourglassScheduler(
            ScheduleRecordListProvider scheduleRecordListProvider,
            TaskFactory taskFactory,
            SchedulerService schedulerService,
            ScheduleUpdateService scheduleUpdateService
    ) {
        return new HourglassScheduler(
                scheduleRecordListProvider,
                taskFactory,
                schedulerService,
                scheduleUpdateService
        );
    }

    @Bean
    public TaskListProvider taskListProvider(ApplicationContext context) {
        return new TaskListProviderImpl(context);
    }

    @Bean
    public NextRunStrategiesFactory nextRunStrategiesFactory() {
        List<NextRunCalcStrategy> nextRunCalcStrategies = List.of(new PeriodicStrategy(),
                new ByCronExpressionStrategy());
        return new NextRunStrategiesFactoryImpl(nextRunCalcStrategies);
    }

    @Bean
    public NextRunModifierFactory nextRunModifierFactory() {
        List<NextRunModifier> modifiers = List.of();
        return new NextRunModifierFactoryImpl(modifiers);
    }

    @Bean
    public ScheduleInfoProcessor scheduleInfoProcessor(
            NextRunStrategiesFactory nextRunStrategiesFactory,
            NextRunModifierFactory nextRunModifierFactory
    ) {
        return new ScheduleInfoProcessor(nextRunStrategiesFactory, nextRunModifierFactory);
    }

    @Bean
    public ScheduleInfoConverter scheduleInfoConverter() {
        return new ScheduleInfoConverterImpl();
    }

    @Bean
    public JobExecutingMonitor jobExecutingMonitor(InstanceId instanceId) {
        JobExecutingMonitor jobExecutingMonitor = new JobExecutingMonitorDummy();

        jobExecutingMonitor.setSchedulerName(instanceId.toString());
        jobExecutingMonitor.setInterval(JOB_EXECUTION_MONITOR_INTERVAL);

        return jobExecutingMonitor;
    }

    @Bean
    public HourglassJugglerSender hourglassJugglerSender(JugglerSender jugglerSender) {
        return new HourglassJugglerSender(jugglerSender);
    }

    @Bean
    public CheckInSchdedulerPingListener checkInSchdedulerPingListener(HourglassJugglerSender hourglassJugglerSender,
                                                                       InstanceId instanceId,
                                                                       InstanceMeta instanceMeta) {
        return new CheckInSchdedulerPingListener(hourglassJugglerSender, instanceId, instanceMeta);
    }

    @Bean
    public SchedulerPingMonitoring schedulerPingMonitoring(List<SchedulerPingListener> schedulerPingListenerList) {
        return new SchedulerPingMonitoring(schedulerPingListenerList);
    }

    @Bean
    public MonitoringWriter monitoringWriter(MetricRegistry metricRegistry,
                                             SchedulerPingMonitoring schedulerPingMonitoring) {
        return new MonitoringWriterImpl(metricRegistry, schedulerPingMonitoring);
    }

    @Bean
    public JobInterceptorsList jobInterceptorsList() {
        return new JobInterceptorsList();
    }

    @Bean
    public TaskFactory taskFactory(
            ApplicationContext context,
            TaskListProvider taskListProvider,
            JobInterceptorsList jobInterceptorsList,
            JobExecutingMonitor jobExecutingMonitor,
            ScheduleInfoConverter scheduleInfoConverter,
            ScheduleInfoProcessor scheduleInfoProcessor
    ) {
        return new TaskFactoryImpl(context, taskListProvider, jobInterceptorsList, jobExecutingMonitor,
                scheduleInfoConverter, scheduleInfoProcessor);
    }

    @Bean
    public ScheduleRecordListProvider scheduleRecordListProvider(
            TaskListProvider taskListProvider,
            ScheduleInfoConverter scheduleInfoConverter,
            TaskParameterizer taskParameterizer,
            JobScheduleInfoFactory jobScheduleInfoFactory
    ) {
        return new ScheduleRecordListProviderPartnerImpl(
                taskListProvider, scheduleInfoConverter, taskParameterizer,
                jobScheduleInfoFactory
        );
    }

    @Bean
    public TaskParameterizer taskParameterizer(ApplicationContext context) {
        Collection<Integer> shardIndexes = List.of();
        return new TaskParameterizer(context, shardIndexes);
    }

    @Bean
    public JobScheduleInfoFactory taskScheduleGetter(ApplicationContext context) {
        return new JobScheduleInfoFactory(context);
    }

    @Bean
    public ThreadsHierarchy threadsHierarchy() {
        return new ThreadsHierarchy();
    }

    @Bean
    public TaskThreadPool taskThreadPool(ThreadsHierarchy threadsHierarchy, HourglassProperties hourglassProperties) {
        return new TaskThreadPoolImpl(hourglassProperties.getThreadPoolSize(), threadsHierarchy);
    }

    @Bean
    public SchedulerInstancePinger schedulerInstancePinger(
            InstanceId instanceId,
            InstanceMeta instanceMeta,
            SchedulerInstancesRepository schedulerInstancesRepository,
            MonitoringWriter monitoringWriter
    ) {
        return new SchedulerInstancePingerImpl(instanceId, JOBS_VERSION, schedulerInstancesRepository, instanceMeta,
                monitoringWriter);
    }

    @Bean
    public RandomChooser<PrimaryId> randomChooser(
            SchedulerInstancesRepository schedulerInstancesRepository,
            InstanceId instanceId
    ) {
        return new RendezvousRandomChooser<>(JOBS_VERSION, schedulerInstancesRepository, instanceId, new MD5Hash());
    }

    @Bean
    @SuppressWarnings("ParameterNumber")
    public SchedulerService schedulerService(
            Storage storage,
            InstanceId instanceId,
            RandomChooser<PrimaryId> randomChooser,
            ThreadsHierarchy threadsHierarchy,
            TaskThreadPool taskThreadPool,
            HourglassProperties hourglassProperties,
            SchedulerInstancePinger schedulerInstancePinger,
            MonitoringWriter monitoringWriter
    ) {
        return new SchedulerServiceImpl(
                storage,
                threadsHierarchy,
                taskThreadPool,
                instanceId,
                randomChooser,
                hourglassProperties,
                schedulerInstancePinger,
                monitoringWriter
        );
    }

    @Bean
    public Storage storage(DSLContext dslContext, InstanceId instanceId, HourglassProperties hourglassProperties) {
        return new StorageImpl(dslContext, instanceId, hourglassProperties, JOBS_VERSION);
    }

    @Bean
    public SchedulerInstancesRepository schedulerInstancesRepository(DSLContext dslContext,
                                                                     HourglassProperties hourglassProperties) {
        return new MysqlSchedulerInstancesRepository.Builder(dslContext)
                .withHeartbeatExpiration(hourglassProperties.getSchedulerHeartbeatExpiration())
                .withLeaderVotingLowerBound(hourglassProperties.getLeaderVotingLowerBound())
                .build();
    }

    @PostConstruct
    private void postConstruct() {
        threadsHierarchy.addUncaughtExceptionHandler((a, b) -> hourglassScheduler.stop());

        try {
            hourglassScheduler.saveSchedule();
            hourglassScheduler.start();
        } catch (Exception ex) {
            LOGGER.error("failed to start scheduler", ex);
            //initializeSchedule может упасть если нет таблиц.
            //В этом случае надо остановить Spring context, иначе приложение продолжит работать(т. к. пулы потоков не
            // daemon'ы)
            springAppTerminator.terminate(1);
        }
    }
}
