package ru.yandex.partner.hourglass.jobs;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.queue.service.TaskQueueService;
import ru.yandex.partner.core.entity.tasks.EmailToProcessingTask;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.libs.annotation.PartnerTransactional;

import static ru.yandex.partner.core.entity.user.model.prop.BaseUserIdPropHolder.ID;
import static ru.yandex.partner.core.entity.user.model.prop.CommonUserLoginPropHolder.LOGIN;
import static ru.yandex.partner.dbschema.partner.Tables.USERS;

@Service
public class EmailToProcessingJobGenerator {
    private static final Logger logger = LoggerFactory.getLogger(EmailToProcessingJobGenerator.class);

    private final UserService userService;
    private final TaskQueueService taskQueueService;
    private final DSLContext dslContext;

    @Autowired
    public EmailToProcessingJobGenerator(UserService userService, TaskQueueService taskQueueService,
                                         DSLContext dslContext) {
        this.userService = userService;
        this.taskQueueService = taskQueueService;
        this.dslContext = dslContext;
    }

    @PartnerTransactional
    public void generateJobs() {

        List<User> users = userService.findAll(QueryOpts.forClass(User.class)
                .withFilter(UserFilters.NEED_TO_EMAIL_PROCESSING.eq(true))
                .withProps(Set.of(ID, LOGIN))
                .withOrder(new OrderBy(ID))
        );

        for (User user : users) {
            try {
                logger.info("Enqueueing EmailToProcessingTask for user {}({})", user.getLogin(), user.getId());
                taskQueueService.enqueue(EmailToProcessingTask.Payload.of(user.getId()));
            } catch (Exception e) {
                logger.error(e.toString());
            }
        }

        dslContext.update(USERS)
                .set(USERS.NEED_TO_EMAIL_PROCESSING, Long.valueOf(0))
                .where(USERS.ID.in(users.stream().map(User::getId).collect(Collectors.toSet())))
                .execute();
    }
}
