package ru.yandex.partner.hourglass.jobs;

import java.time.Duration;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.scheduling.support.CronSequenceGenerator;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.partner.defaultconfiguration.logging.ErrorBoosterMessageConverter;


public class PartnerJobUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(PartnerJobUtils.class);
    private static final Duration EVENTS_TIMEOUT = Duration.ofMinutes(10);

    private PartnerJobUtils() {
    }

    public static void putToMDC(String cronName) {
        MDC.put(ErrorBoosterMessageConverter.MDCKeys.CRON_PATH, "Hourglass");
        MDC.put(ErrorBoosterMessageConverter.MDCKeys.CRON_METHOD, cronName);
        LOGGER.info("Started cron {}", cronName);
    }

    public static Long getCheckTtlByJobClass(Class<? extends HourglassJob> jobClass) {
        if (!jobClass.isAnnotationPresent(Hourglass.class)) {
            throw new IllegalArgumentException("Job must be annotated with @Hourglass!");
        }
        var hourglass = jobClass.getAnnotation(Hourglass.class);
        // TODO: переписать на org.springframework.scheduling.support.CronExpression
        // когда обновим спринг до >= 5.3
        CronSequenceGenerator generator = new CronSequenceGenerator(hourglass.cronExpression());
        Date nextDate = generator.next(new Date());
        Date afterNext = generator.next(nextDate);
        long diffInMillis = afterNext.getTime() - nextDate.getTime();
        long secondsDiff = TimeUnit.SECONDS.convert(diffInMillis, TimeUnit.MILLISECONDS);
        return calculateCheckTtl(secondsDiff);
    }

    /**
     * https://github.yandex-team.ru/partner/partner2/blob/5c573b14aa31c7ccb601c9c46e9038dcbe2f175e/lib/Utils/Cron
     * .pm#L27
     *
     * @return ttl после проверки всех условий
     */
    public static Long calculateCheckTtl(Long seconds) {
        // увеличиваем на 20%, но не больше чем на сутки
        seconds = seconds + Math.min((long) (0.2 * seconds), Duration.ofDays(1).toSeconds());

        // и не меньше часа
        seconds = Math.max(Duration.ofHours(1).toSeconds(), seconds);

        // juggler не позволяет указать ttl > 10 дней
        seconds = Math.min(Duration.ofDays(10).toSeconds(), seconds);

        return seconds;
    }
}
