package ru.yandex.partner.hourglass.jobs;

import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.partner.hourglass.monitoring.Alertable;
import ru.yandex.partner.hourglass.monitoring.JugglerCheckFactory;
import ru.yandex.partner.hourglass.service.ExecutionHelperService;
import ru.yandex.partner.libs.juggler.JugglerCheck;

import static ru.yandex.partner.hourglass.jobs.PartnerJobUtils.getCheckTtlByJobClass;

@ParametersAreNonnullByDefault
public abstract class PartnersParamsJob<T> extends DirectParameterizedJob<T> implements PartnersJob, Alertable {
    private static final Logger LOGGER = LoggerFactory.getLogger(PartnersParamsJob.class);
    private final ExecutionHelperService executionHelperService;
    private final ApplicationContext applicationContext;

    protected PartnersParamsJob(ExecutionHelperService executionHelperService, ApplicationContext applicationContext) {
        this.executionHelperService = executionHelperService;
        this.applicationContext = applicationContext;
    }

    @Override
    public final void execute() {
        executionHelperService.execute(this);
}


    @Override
    public String getCronName() {
        // param будет выдан задаче в рантайме директовским шедулером
        return getCronName(this.getParam());
    }

    private String getCronName(String param) {
        return getBaseName() + "_" + param;
    }

    /**
     * Имя таски-шаблона
     */
    private String getBaseName() {
        return this.getClass().getSimpleName();
    }

    /**
     * Все имена кронов с параметром
     * @return
     */
    public List<String> getAllCronNames() {
        return Optional.of(this.getClass())
                .map(c -> c.getAnnotation(ParameterizedBy.class))
                .map(a -> getAllParams(a.parametersSource(), applicationContext))
                .map(l -> l.stream().map(this::getCronName).toList())
                .orElseThrow(() ->
                        new RuntimeException("Parameterized job %s isn't annotated with @ParameterizedBy!"
                                .formatted(this.getClass().getSimpleName()))
                );
    }

    @Override
    public List<JugglerCheck> getJugglerChecks(JugglerCheckFactory jugglerCheckFactory) {
        Long ttl = getCheckTtlByJobClass(this.getClass());
        return jugglerCheckFactory.createChecks(getBaseName(), ttl, getAllCronNames(), getJugglerWikiUrl());
    }

    public abstract JobProcedure getJobCallback();
}
