package ru.yandex.partner.hourglass.jobs.bk;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.ClassPathResource;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.partner.hourglass.jobs.JobProcedure;
import ru.yandex.partner.hourglass.jobs.SimplePartnersJob;
import ru.yandex.partner.hourglass.service.ExecutionHelperService;
import ru.yandex.partner.hourglass.service.SolomonService;
import ru.yandex.partner.hourglass.yql.YqlService;

/**
 * Джоба умеет работать только на проде, так как оттуда есть доступ до продовых табличек с нужными данными,
 * чтобы просчитать условие недоезда. На тестовых кластерах этих таблиц нет
 */
@Profile("prod")
@Hourglass(cronExpression = "*/15 * * * * *")
public class ArriveToBkCheckJob extends SimplePartnersJob {

    private final YqlService yqlService;
    private final SolomonService solomonService;
    private static final String YQL_QUERY_PATH = "yql/select_not_arrived_page_ids.yql";
    private final String yqlQuery;

    @Autowired
    public ArriveToBkCheckJob(ExecutionHelperService executionHelperService,
                              YqlService yqlService, SolomonService solomonService) {
        super(executionHelperService);
        this.yqlService = yqlService;
        this.solomonService = solomonService;
        try {
            this.yqlQuery = IOUtils.toString(new ClassPathResource(YQL_QUERY_PATH).getInputStream());
        } catch (IOException e) {
            throw new RuntimeException("Can't read resource from %s".formatted(YQL_QUERY_PATH), e);
        }
    }

    @Override
    public JobProcedure getJobCallback() {
        RowMapper<Long> mapper = (rs, rowNum) -> Long.parseLong(rs.getString("page_id"));
        return () -> {
            List<Long> pageIds = yqlService.query(yqlQuery, mapper);

            if (pageIds.isEmpty()) {
                this.setJugglerStatus(JugglerStatus.OK, "All pages arrived to BK");
            } else {
                String idsDescription = pageIds.stream()
                        .limit(10)
                        .map(Object::toString)
                        .collect(Collectors.joining(", "))
                        .concat("...");
                this.setJugglerStatus(
                        JugglerStatus.CRIT,
                        "[ %s ] pages in total didn't make it to BK. IDs: %s".formatted(
                                pageIds.size(), idsDescription
                        )
                );
            }
            solomonService.sendToSolomon(
                    Map.of(
                            "type", "cron",
                            "sensor", "not_arrived_pages_count",
                            "cron_path", this.getCronName()
                    ),
                    pageIds.size()
            );
        };
    }

    @Override
    public String getJugglerWikiUrl() {
        return "https://wiki.yandex-team.ru/partner/w/devops/monitoring/juggler-checks/proverkadoezdaploshhadkidobk/";
    }
}
