package ru.yandex.partner.hourglass.monitoring;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.Nullable;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import ru.yandex.partner.hourglass.configuration.property.JugglerProps;
import ru.yandex.partner.libs.juggler.CheckGroup;
import ru.yandex.partner.libs.juggler.CheckMeta;
import ru.yandex.partner.libs.juggler.CheckMetaUrl;
import ru.yandex.partner.libs.juggler.JugglerCheck;

@Component
public class JugglerCheckFactory {

    private final JugglerProps properties;
    private final List<String> allServices;
    private final Environment environment;

    public JugglerCheckFactory(JugglerProps properties, Environment environment) {
        this.properties = properties;
        this.environment = environment;
        this.allServices = new ArrayList<>();
    }

    /**
     * Factory method для простой проверки без детей (например SimplePartnersJob, у которой 1 тред)
     * @param parentName
     * @param ttl
     * @return
     */
    public List<JugglerCheck> createChecks(String parentName, Long ttl) {
        return createChecks(parentName, ttl, List.of());
    }

    public List<JugglerCheck> createChecks(String parentName, Long ttl, String wikiUrl) {
        return createChecks(parentName, ttl, List.of(), wikiUrl, Set.of());
    }

    /**
     * Factory method для проверки агрегирующей проверки с детьми. В результате массив будет childChecks + parentCheck.
     * Пример (ValidationJob, работает в несколько тредов; на каждый тред своя дочерняя проверка)
     * @param parentName
     * @param ttl
     * @param childNames
     * @return
     */
    public List<JugglerCheck> createChecks(String parentName, Long ttl, List<String> childNames) {
        return createChecks(parentName, ttl, childNames, null, Set.of());
    }

    public List<JugglerCheck> createChecks(String parentName, Long ttl, Set<JugglerCheckTag> tagExceptions) {
        return createChecks(parentName, ttl, List.of(), null, tagExceptions);
    }

    public List<JugglerCheck> createChecks(
            String parentName,
            Long ttl,
            List<String> childNames,
            @Nullable String wikiUrl
    ) {
        return createChecks(parentName, ttl, childNames, wikiUrl, Set.of());
    }

    public List<JugglerCheck> createChecks(
            String parentName,
            Long ttl,
            List<String> childNames,
            @Nullable String wikiUrl,
            Set<JugglerCheckTag> tagExceptions
    ) {
        List<JugglerCheck> checks = childNames.stream()
                .map(serviceName -> newCheckBuilder(serviceName, ttl, wikiUrl)
                        .addTags(JugglerCheckTag.getChildCheckTags(environment))
                        .build()
                )
                .collect(Collectors.toList());

        List<CheckGroup> checkGroups = childNames.stream()
                .map(this::newCheckGroup).toList();

        JugglerCheck.Builder parentBuilder = newCheckBuilder(parentName, ttl, wikiUrl);
        if (!childNames.isEmpty()) {
            parentBuilder.setAggregator("logic_or");
            parentBuilder.setChildren(checkGroups);
        }
        JugglerCheck parentCheck = parentBuilder
                .addTags(JugglerCheckTag.getTagsExcept(environment, tagExceptions))
                .setPronounce(properties.getPronounceTemplate().formatted(properties.getHostName(), parentName))
                .build();
        checks.add(parentCheck);
        allServices.add(parentName);

        return checks;
    }

    /**
     * Агрегат для всех ранее созданных проверок
     * @return
     */
    public JugglerCheck createMetaCheck() {
        List<CheckGroup> checkGroups = allServices.stream()
                .sorted()
                .map(this::newCheckGroup).toList();

        return newCheckBuilder(
                properties.getMetaAggregator().getServiceName(),
                properties.getMetaAggregator().getTtl().toSeconds(),
                null)
                .setAggregator("logic_or")
                .setChildren(checkGroups)
                .addTags(JugglerCheckTag.getAllTags(environment))
                .build();
    }

    private JugglerCheck.Builder newCheckBuilder(String service, Long ttl, String wikiUrl) {
        return new JugglerCheck.Builder()
                .setService(service)
                .setTtl(ttl)
                .setMeta(newCheckMeta(wikiUrl))
                .setProject(properties.getProject())
                .setHost(properties.getHostName())
                .setRefreshTime(properties.getRefreshTime().toSeconds());
    }

    private CheckGroup newCheckGroup(String service) {
        return new CheckGroup.Builder()
                .setService(service)
                .setInstance(null)
                .setType("HOST")
                .setHost(properties.getHostName())
                .build();
    }

    private CheckMeta newCheckMeta(String wikiUrl) {
        return new CheckMeta(List.of(newCheckMetaUrl(wikiUrl)));
    }

    private CheckMetaUrl newCheckMetaUrl(String wikiUrl) {
        return new CheckMetaUrl(
                properties.getMeta().getTitle(),
                wikiUrl != null ? wikiUrl : properties.getMeta().getUrl(),
                "wiki"
        );
    }
}
