package ru.yandex.partner.hourglass.monitoring;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import org.springframework.core.env.Environment;
import org.springframework.core.env.Profiles;

public enum JugglerCheckTag {
    JAVA("java", false),
    STANDARD_NOTIFICATION("standard-notification", Profiles.of("!dev")),
    SHOW_ON_DASHBOARD("show-on-dashboard", sameAs(STANDARD_NOTIFICATION));

    private final String name;

    // если включен, то тег не будет докидываться на дочерние агрегаты, когда они имеют место.
    // Например, проверка создается для параметризованной джобы
    private final boolean isParentOnly;

    // предикат над спринговыеми профилями, который определит - добавлять тег в проверки или нет
    // пример: не хотим видеть на дешборде дежурного проверки с dev-а
    private final Profiles allowedProfiles;

    JugglerCheckTag(String name) {
        this(name, true);
    }

    JugglerCheckTag(String name, Boolean isParentOnly) {
        this(name, isParentOnly, Profiles.of("default"));
    }

    JugglerCheckTag(String name, Profiles allowedProfiles) {
        this(name, true, allowedProfiles);
    }

    JugglerCheckTag(String name, boolean isParentOnly, Profiles allowedProfiles) {
        this.name = name;
        this.isParentOnly = isParentOnly;
        this.allowedProfiles = allowedProfiles;
    }

    public static List<String> getChildCheckTags(Environment environment) {
        return filteredByEnv(environment)
                .filter(t -> !t.isParentOnly)
                .map(JugglerCheckTag::toString).toList();
    }

    public static List<String> getAllTags(Environment environment) {
        return filteredByEnv(environment)
                .map(JugglerCheckTag::toString).toList();
    }

    public static List<String> getTagsExcept(Environment environment, Set<JugglerCheckTag> exceptions) {
        return filteredByEnv(environment)
                .filter(t -> !exceptions.contains(t))
                .map(JugglerCheckTag::toString).toList();
    }

    private static Stream<JugglerCheckTag> filteredByEnv(Environment environment) {
        return Arrays.stream(JugglerCheckTag.values())
                .filter(t -> environment.acceptsProfiles(t.allowedProfiles));
    }
    @Override
    public String toString() {
        return name;
    }

    private static Profiles sameAs(JugglerCheckTag tag) {
        return tag.allowedProfiles;
    }
}
