package ru.yandex.partner.hourglass.monitoring;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.jooq.DSLContext;
import org.springframework.scheduling.annotation.Scheduled;

import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.partner.hourglass.service.JugglerEventService;
import ru.yandex.partner.libs.juggler.JugglerCheck;

import static java.time.LocalDateTime.now;
import static ru.yandex.partner.dbschema.partner.Tables.SCHEDULER_INSTANCES;


public class SchedulerInstancesMonitoring implements Alertable {

    private final DSLContext dslContext;
    private final JugglerEventService jugglerEventService;
    private final SchedulerInstancesProps schedulerInstancesProps;

    public SchedulerInstancesMonitoring(DSLContext dslContext, JugglerEventService jugglerEventService,
                                        SchedulerInstancesProps schedulerInstancesProps) {
        this.dslContext = dslContext;
        this.jugglerEventService = jugglerEventService;
        this.schedulerInstancesProps = schedulerInstancesProps;
    }

    /**
     * Проверяет работоспособность шедулера, что hourglass job-ы запускаются.
     * Регулярное исполнение метода обеспечивается спрингом, т.к мы не можем
     * использовать хаургласс для мониторинга его же самого, ведь если сломается хаургласс,
     * то мониторинг за ним :)
     */
    @Scheduled(fixedRateString = "#{@schedulerInstancesProps.rate.toSeconds()}", timeUnit = TimeUnit.SECONDS)
    void checkAliveScheduledInstances() {
        long aliveWindowSeconds = schedulerInstancesProps.getAliveWindow().toSeconds();
        Integer count = dslContext.selectCount().from(SCHEDULER_INSTANCES)
                .where(SCHEDULER_INSTANCES.HEARTBEAT_TIME.ge(now().minusSeconds(aliveWindowSeconds)))
                .fetchOne().value1();

        JugglerStatus status;
        String description;
        if (count < 0) {
            status = JugglerStatus.CRIT;
            description = "Hourglass is not working: no alive scheduled instances found " +
                    "in the scheduled_instances table";
        } else {
            status = JugglerStatus.OK;
            description = "Hougrlass is working properly, found %s alive instances in the scheduled_instances table"
                    .formatted(count);
        }
        jugglerEventService.sendEvent(
                getClass().getSimpleName(),
                status,
                description
        );
    }

    @Override
    public List<JugglerCheck> getJugglerChecks(JugglerCheckFactory jugglerCheckFactory) {
        return jugglerCheckFactory.createChecks(
                getClass().getSimpleName(),
                schedulerInstancesProps.getTtl().toSeconds(),
                "https://wiki.yandex-team.ru/partner/w/devops/incident-solutions#hourglassdzhobygorjatnodatae.g" +
                        ".rtbblockvalidationjob"
        );
    }
}
