package ru.yandex.partner.hourglass.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.beust.jcommander.Parameter;
import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.boot.SpringApplication;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.scheduler.support.ParametersSource;
import ru.yandex.direct.scheduler.support.PeriodicJobWrapper;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;

/**
 * Класс, позволяющий запустить одну конкретную задачу.
 * Требуется в основном для запуска при отладке.
 * Запускать необходимо с дополнительным профилeм debugjob
 * Класс требуемой задачи нужно указать первым аргументом через -jc, --jobclass.
 * Для параметризованных джоб нужно обязательно указазать параметр через -p, --param.
 */
@Component
@Profile("debugjob")
public class DebugJobRunner implements CommandLineRunner {
    private static final Logger logger = LoggerFactory.getLogger(DebugJobRunner.class);
    private final List<BaseDirectJob> jobs;
    private final ApplicationContext ctx;

    @Autowired
    public DebugJobRunner(List<BaseDirectJob> jobs, ApplicationContext applicationContext) {
        this.jobs = jobs;
        this.ctx = applicationContext;
    }

    @Override
    public void run(String... args) throws Exception {
        logger.info("debug job runner started");
        List<String> argList = Arrays.asList(args);
        String[] filteredArgs = argList.stream().filter(elt -> !elt.contains("=")).toArray(String[]::new);
        var runnerParams = new RunnerParams();
        ParserWithHelp.parse(DebugJobRunner.class.getCanonicalName(), filteredArgs, runnerParams);
        this.runJob(runnerParams);
        System.exit(SpringApplication.exit(ctx, () -> 0));
    }

    public void runJob(RunnerParams runnerParams) {
        runnerParams.validate();

        Class<? extends BaseDirectJob> clazz = getJobClass(runnerParams.jobClassNames.get(0));
        logger.info("Found job {}", clazz.getCanonicalName());
        if (DirectParameterizedJob.isParameterized(clazz)) {
            ParameterizedBy parameterizedBy = clazz.getAnnotation(ParameterizedBy.class);
            Class<? extends ParametersSource> parametersSourceClazz = parameterizedBy.parametersSource();
            Set<String> allParams = new HashSet<>(DirectParameterizedJob.getAllParams(parametersSourceClazz, ctx));
            if (!runnerParams.parameters.isEmpty()) {
                for (String param : runnerParams.parameters) {
                    if (!allParams.contains(param)) {
                        logger.error(
                                "Param '{}' is not contained in all available parameters '{}', job execution will be " +
                                        "skipped",
                                param, allParams);
                        continue;
                    }
                    logger.info("executing with param {}", param);
                    PeriodicJobWrapper job = new PeriodicJobWrapper(ctx.getBean(clazz));
                    job.execute(TaskParametersMap.of(DirectParameterizedJob.PARAM_NAME, param));
                }
            }
        } else {
            PeriodicJobWrapper job = new PeriodicJobWrapper(ctx.getBean(clazz));
            job.execute(TaskParametersMap.of());
        }
    }

    /**
     * ищем в контексте Job по имени класса
     */
    private Class<? extends BaseDirectJob> getJobClass(String jobClassName) {
        List<BaseDirectJob> matchedJobs = jobs.stream()
                .filter(j ->
                        j.getClass().getCanonicalName().equals(jobClassName)
                                || j.getClass().getCanonicalName().endsWith("." + jobClassName)
                )
                .collect(toList());

        checkState(!matchedJobs.isEmpty(), "Job bean is not found: %s", jobClassName);
        checkState(matchedJobs.size() == 1, "More than one job bean found");

        return matchedJobs.get(0).getClass();
    }

    /**
     * параметры приложения, понимаемые JobRunner
     */
    public static class RunnerParams {
        @Parameter(
                names = {"-jc", "--jobclass"},
                description = "Class name of job to run",
                required = true
        )
        private List<String> jobClassNames = new ArrayList<>();

        @Parameter(
                names = {"-p", "--param"},
                description = "Parameter value"
        )
        private List<String> parameters = new ArrayList<>();

        public RunnerParams() {

        }

        public RunnerParams(List<String> jobClassNames, List<String> parameters) {
            this.jobClassNames = jobClassNames;
            this.parameters = parameters;
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("jobClassNames", jobClassNames)
                    .add("parameters", parameters)
                    .toString();
        }

        void validate() {
            checkArgument(jobClassNames.size() == 1, "Supported only one job class");
        }
    }
}
