package ru.yandex.partner.hourglass.service;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.partner.hourglass.jackson.Metric;
import ru.yandex.partner.hourglass.jackson.SolomonRequest;


@Service
@ConditionalOnProperty(name = "solomon.send", havingValue = "true")
public class SolomonServiceImpl implements SolomonService {

    private static final Logger LOGGER = LoggerFactory.getLogger(SolomonServiceImpl.class);
    private static final String BAD_REQUEST_MSG_TEMPLATE = "An error occurred while sending metric to solomon:\n " +
            "Code: %s, %s";
    private static final String SERIALIZATION_ERROR_MSG_TEMPLATE =
            """
            An error occurred while attempt to serialize cron metric:
            %s
            %s""";

    private final ObjectMapper objectMapper;
    private final SolomonRequestExecutor solomonRequestExecutor;

    public SolomonServiceImpl(
            ObjectMapper objectMapper,
            SolomonRequestExecutor solomonRequestExecutor) {
        this.solomonRequestExecutor = solomonRequestExecutor;
        this.objectMapper = objectMapper;
    }

    @Override
    public HttpStatus sendToSolomon(Map<String, String> labels, long value) {
        var data = new SolomonRequest(List.of(new Metric(labels, value)));

        try {
            var jsonData = objectMapper.writeValueAsString(data);
            var result = solomonRequestExecutor.pushToSolomon(jsonData);
            if (!result.getStatusCode().is2xxSuccessful()) {
                LOGGER.error(String.format(BAD_REQUEST_MSG_TEMPLATE, result.getStatusCode(),
                        result.getBody()));
            }
            return result.getStatusCode();
        } catch (Exception e) {
            LOGGER.error(String.format(SERIALIZATION_ERROR_MSG_TEMPLATE, e.getMessage(), e));
        }
        return HttpStatus.INTERNAL_SERVER_ERROR;
    }
}
