package ru.yandex.partner.intapi;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.jgonian.ipmath.AbstractIp;
import com.github.jgonian.ipmath.AbstractIpRange;
import com.github.jgonian.ipmath.Ipv4;
import com.github.jgonian.ipmath.Ipv4Range;
import com.github.jgonian.ipmath.Ipv6;
import com.github.jgonian.ipmath.Ipv6Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Класс утилит для приложения
 */
@ParametersAreNonnullByDefault
public class Utils {
    private static final Logger LOGGER = LoggerFactory.getLogger(Utils.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private Utils() {
    }

    /**
     * Парсит json и приводит к типу T
     *
     * @param json строка с json
     * @param type TypeReference типа T для распарсинга
     * @param <T>  тип для приведения
     * @return Объект типа type
     */

    public static <T> T parseTypeFromJson(@Nonnull String json, @Nonnull TypeReference<T> type) {
        try {
            return OBJECT_MAPPER.readValue(json, type);
        } catch (IOException e) {
            String warning = String.format("Error parsing JSON string : %s to type: %s", json,
                    type.getType().toString());
            LOGGER.warn(warning);
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Возвращает IP4/IP6 адрес из строки, если её можно распарсить в виде IP адреса
     *
     * @param addr адрес в виде строки
     * @return Ipv4/Ipv6 адрес или null, если строку распарсить не получилось
     */
    public static @Nullable
    AbstractIp getIpFromString(@Nullable String addr) {
        if (addr == null) {
            return null;
        }
        try {
            InetAddress address = InetAddress.getByName(addr);
            if (address instanceof Inet4Address) {
                return Ipv4.parse(addr);
            } else if (address instanceof Inet6Address) {
                return Ipv6.parse(addr);
            } else {
                return null;
            }
        } catch (UnknownHostException e) {
            return null;
        }
    }

    /**
     * Проверяет вхождение конкретного IP адреса в список подсетей
     *
     * @param ip           IP адрес типа Ipv4/Ipv6
     * @param netRangeList список подсетей типа Ipv4Range/Ipv6Range
     * @return true, если IP входит в какую-то из сетей, false, если ни одного вхождения не найдено
     */
    public static boolean isIpInNetRanges(AbstractIp ip, List<AbstractIpRange> netRangeList) {
        for (AbstractIpRange range : netRangeList) {
            if (ip instanceof Ipv4 && range instanceof Ipv4Range &&
                    ((Ipv4Range) range).contains((Ipv4) ip)) {
                return true;
            }
            if (ip instanceof Ipv6 && range instanceof Ipv6Range &&
                    ((Ipv6Range) range).contains((Ipv6) ip)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Разбирает строку json на список строк
     *
     * @param s входная json строка
     * @return список строк
     */
    public static List<String> jsonStringToList(@Nullable String s) {
        if (s == null) {
            return Collections.emptyList();
        }
        return parseTypeFromJson(s, new TypeReference<List<String>>() {
        });
    }

    /**
     * Преобразует список подсетей из списка строк в список типа Ipv4Range/Ipv6Range
     *
     * @param netList входной список строк
     * @return список подсетей
     */
    public static List<AbstractIpRange> netListToRanges(List<String> netList) {
        List<AbstractIpRange> netRangeList = new ArrayList<>();
        for (String netName : netList) {
            try {
                netRangeList.add(Ipv4Range.parse(netName));
            } catch (IllegalArgumentException e4) {
                try {
                    netRangeList.add(Ipv6Range.parse(netName));
                } catch (IllegalArgumentException e6) {
                    LOGGER.warn("Exception on NET range parse of {}", netName);
                }
            }
        }
        return netRangeList;
    }

    /**
     * Преобразует json строку в список подсетей типа Ipv4Range/Ipv6Range
     *
     * @param s входная json строка
     * @return список подсетей
     */
    public static List<AbstractIpRange> jsonStringToRanges(@Nullable String s) {
        return netListToRanges(jsonStringToList(s));
    }

    /**
     * Проверяет является ли переданный IP адрес адресом локального хоста (127.x.x.x или 0.0.0.0.0.0.0.1)
     *
     * @param addr строка с IP адресом
     * @return true, если является локальным хостом, false, если нет
     */
    public static boolean isLocalhost(@Nullable String addr) {
        if (addr == null) {
            return false;
        }
        try {
            InetAddress address = InetAddress.getByName(addr);
            return address.isLoopbackAddress();
        } catch (UnknownHostException e) {
            return false;
        }
    }

    /**
     * Пытается получить реальный IP, с учётом хедеров X-Real-IP и X-Forwarded-For
     *
     * @param request http запрос
     * @return строка с IP адресом
     */
    public static String getIpAddrFromRequest(HttpServletRequest request) {
        String ip = request.getHeader("X-Real-IP");
        if (ip != null && !"".equals(ip.trim()) && !"unknown".equalsIgnoreCase(ip)) {
            return ip;
        }
        return request.getRemoteAddr();
    }
}
