package ru.yandex.partner.intapi.acl;

import java.util.List;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.github.jgonian.ipmath.AbstractIp;
import com.github.jgonian.ipmath.AbstractIpRange;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.intapi.Utils;

/**
 * Сервисный класс ACL
 */
@Service
public class ACLService {
    private static final Logger LOGGER = LoggerFactory.getLogger(ACLService.class);
    private final IACLRepository aclRepository;
    private final ACLConfig aclConfig;

    @Autowired
    public ACLService(IACLRepository aclRepository, ACLConfig aclConfig) {
        this.aclRepository = aclRepository;
        this.aclConfig = aclConfig;
    }

    /**
     * Проверяет доступ с адреса к текущему пути
     *
     * @param aclRequestPath ACL путь для доступа
     * @param remoteAddr     Адрес пользователя
     * @return true, если доступ разрешён, false, если запрещён
     * @throws ExecutionException исключение при получении данных
     */
    public boolean checkACL(@Nonnull ACLRequestPath aclRequestPath, @Nullable String remoteAddr)
            throws ExecutionException {

        if (aclConfig.isAllowLocalhost() && Utils.isLocalhost(remoteAddr)) {
            return true;
        }

        AbstractIp ip = Utils.getIpFromString(remoteAddr);

        if (ip == null) {
            return false;
        }

        List<AbstractIpRange> netRangeList = aclRepository.getACL(aclRequestPath);

        if (Utils.isIpInNetRanges(ip, netRangeList)) {
            return true;
        } else {
            LOGGER.warn("Access denied from {}", remoteAddr);
            return false;
        }
    }
}
