package ru.yandex.partner.intapi.configuration;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.escape.Escaper;
import com.google.common.escape.Escapers;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.converter.AbstractHttpMessageConverter;
import org.springframework.http.converter.HttpMessageNotWritableException;

import ru.yandex.partner.intapi.queryresult.QueryResult;

import static com.fasterxml.jackson.databind.PropertyNamingStrategy.SNAKE_CASE;

/**
 * Сериализатор QueryResult в tsv.
 * Сериализация по правилам Партнёрского интерфейса - значения не кавычатся, в конце выводим маркер.
 */
@ParametersAreNonnullByDefault
public class QueryResultTsvMessageConverter extends AbstractHttpMessageConverter<Object> {
    private static final String END_MARKER = "#END";

    private static final Escaper TSV_ESCAPER = Escapers.builder()
            .addEscape('\\', "\\\\")
            .addEscape('\f', "\\f")
            .addEscape('\n', "\\n")
            .addEscape('\r', "\\r")
            .addEscape('\t', "\\t")
            .build();

    private static final ObjectMapper MAPPER = new ObjectMapper().setPropertyNamingStrategy(SNAKE_CASE);
    private static final TypeReference<Map<String, String>> MAP_TYPE_REFERENCE = new TypeReference<>() {
    };

    public QueryResultTsvMessageConverter() {
        super(WebConfig.MEDIA_TYPE_TSV);
    }

    @Override
    protected boolean supports(Class<?> clazz) {
        return QueryResult.class.isAssignableFrom(clazz);
    }

    @Override
    @Nonnull
    protected Object readInternal(Class<?> clazz, HttpInputMessage inputMessage) {
        throw new UnsupportedOperationException("Can't read data in tsv format");
    }

    @Override
    protected void writeInternal(
            Object o, HttpOutputMessage outputMessage
    ) throws IOException, HttpMessageNotWritableException {
        if (!(o instanceof QueryResult)) {
            throw new IllegalStateException("Supported only QueryResult, but argument is " + o.getClass().getName());
        }
        QueryResult<?> qr = (QueryResult<?>) o;

        try (
                var outputStream = new BufferedOutputStream(outputMessage.getBody());
                var writer = new PrintWriter(outputStream);
        ) {

            String[] fields = qr.getFields().toArray(new String[0]);

            if (qr.getOptions().getWithHeaders() != null && qr.getOptions().getWithHeaders()) {
                writeLine(writer, fields);
            }

            for (Object obj : qr.getData()) {
                writeLine(writer, getValuesArray(obj, fields));
            }

            writer.write(END_MARKER);
            writer.write("\n");
        }
    }

    /*
     * Есть разные варианты получить поля:
     * - голый рефлекшен (медленно, некрасиво, проблема с snake_case)
     * - common-beanutil (медленно, проблема со snake_case)
     * Выбираем jackson, как самый простой, с возможность настройки имён полей и быстрый, если выводятся многие поля
     */
    private String[] getValuesArray(Object obj, String[] fields) {
        Map<String, String> map = MAPPER.convertValue(obj, MAP_TYPE_REFERENCE);
        String[] vals = new String[fields.length];
        for (int i = 0; i < fields.length; i++) {
            vals[i] = map.get(fields[i]);
        }
        return vals;
    }

    private void writeLine(PrintWriter writer, String[] vals) {
        for (int i = 0; i < vals.length; i++) {
            if (i != 0) {
                writer.print('\t');
            }
            writer.print(tsvEscape(vals[i]));
        }
        writer.print('\n');
    }

    @VisibleForTesting
    String tsvEscape(@Nullable String s) {
        if (s == null) {
            return "";
        }

        return TSV_ESCAPER.escape(s);
    }
}
