package ru.yandex.partner.intapi.configuration;

import java.util.List;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.servlet.config.annotation.ContentNegotiationConfigurer;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.PathMatchConfigurer;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import ru.yandex.partner.defaultconfiguration.GracefulShutdown;
import ru.yandex.partner.intapi.acl.ACLInterceptor;

/**
 * Класс для дополнительной настройки WebMVC
 */
@Configuration
@Import({GracefulShutdown.class})
@EnableWebMvc
public class WebConfig implements WebMvcConfigurer {
    public static final String CONTENT_TYPE_TSV = "text/tab-separated-values";
    public static final MediaType MEDIA_TYPE_TSV = MediaType.valueOf(CONTENT_TYPE_TSV);

    /**
     * Нужен для решения проблемы
     * с interceptor, содержащим @Autowired поле
     * https://stackoverflow.com/questions/23349180/java-config-for-spring-interceptor-where-interceptor-is-using
     * -autowired-spring-b
     *
     * @return экземпляр ACLInterceptor
     */
    @Bean
    public ACLInterceptor aclInterceptor() {
        return new ACLInterceptor();
    }

    /**
     * Настройка согласования типов возвращаемого контента:
     * - возвращаемые типы json/xml
     * - возвращаемый по умолчанию тип json
     * - определение типа возвращаемого контента по суффиксу URL разрешено
     * -- пример запроса: ../page_name.xml - вернёт xml
     * - определение типа возвращаемого контента по параметрам запроса разрешено
     * - название параметра для определения типа контента format
     * -- пример запроса: ../page_name?format=xml - вернёт xml
     *
     * @param configurer ContentNegotiationConfigurer
     */
    @Override
    public void configureContentNegotiation(ContentNegotiationConfigurer configurer) {
        configurer
                .mediaType("json", MediaType.APPLICATION_JSON)
                .mediaType("xml", MediaType.APPLICATION_XML)
                .mediaType("tsv", MEDIA_TYPE_TSV)
                .favorPathExtension(true)
                .favorParameter(true)
                .parameterName("format")
                .ignoreAcceptHeader(true)
                .defaultContentType(MediaType.APPLICATION_JSON);
    }

    @Override
    public void configurePathMatch(PathMatchConfigurer configurer) {
        configurer.setUseSuffixPatternMatch(true);
    }

    /**
     * Регистрирует interceptor
     *
     * @param registry InterceptorRegistry
     */
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(aclInterceptor());
    }

    @Bean
    QueryResultTsvMessageConverter queryResultTsvMessageConverter() {
        return new QueryResultTsvMessageConverter();
    }

    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        converters.add(queryResultTsvMessageConverter());
    }
}
