package ru.yandex.partner.intapi.acl;

import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.test.web.servlet.MockMvc;

import ru.yandex.partner.intapi.App;
import ru.yandex.partner.intapi.antifrod.DbQueryService;
import ru.yandex.partner.test.db.MockedMysqlTestConfiguration;

import static org.mockito.Mockito.atMost;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static ru.yandex.partner.dbschema.partner.tables.IntapiAcl.INTAPI_ACL;

@SpringBootTest(
        properties = "spring.main.allow-bean-definition-overriding=true",
        classes = {App.class, MockedMysqlTestConfiguration.class}
)
@AutoConfigureMockMvc
public class AclCacheTest {
    @Autowired
    DSLContext dsl;

    @Autowired
    DbQueryService dbQueryService;

    @Autowired
    MockMvc mvc;

    @Autowired
    ACLConfig aclConfig;

    @SpyBean
    ACLRepository aclRepository;

    @Test
    void checkAclCachingWorks() throws Exception {
        aclConfig.setAllowLocalhost(false);

        aclRepository.clearACLCache();

        String antifrodPath = "antifrod";
        String reportMethod = "page_owner_report";

        mockRange(antifrodPath, reportMethod, "[\"127.0.0.1/32\",\"0:0:0:0:0:0:0:1/128\"]");

        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isOk());
        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isOk());
        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isOk());
        verify(aclRepository, atMost(1))
                .getACL(new ACLRequestPath(antifrodPath, reportMethod));

        Mockito.reset(aclRepository);
        Mockito.reset(dsl);
        aclRepository.clearACLCache();
        mockRange(antifrodPath, reportMethod, null);
        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isForbidden());
        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isForbidden());
        mvc.perform(get("/intapi/antifrod/page_owner_report")).andExpect(status().isForbidden());
        verify(aclRepository, atMost(1))
                .getACL(new ACLRequestPath(antifrodPath, reportMethod));
    }

    private void mockRange(String path, String method, String range) {
        SelectConditionStep<Record1<String>> select = Mockito.mock(SelectConditionStep.class);

        when(dsl.select(INTAPI_ACL.ACL_CACHED)
                .from(INTAPI_ACL)
                .where(INTAPI_ACL.PATH.eq(path))
                .and(INTAPI_ACL.METHOD.eq(method))
        ).thenReturn(select);

        when(select.fetchOne(INTAPI_ACL.ACL_CACHED)).thenReturn(range);
    }
}
