package ru.yandex.partner.intapi.antifrod;

import java.io.IOException;

import com.fasterxml.jackson.core.type.TypeReference;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;

import ru.yandex.partner.intapi.IntApiTestUtils;
import ru.yandex.partner.intapi.acl.ACLConfig;
import ru.yandex.partner.intapi.acl.IACLRepository;
import ru.yandex.partner.intapi.configuration.IntapiTest;
import ru.yandex.partner.intapi.queryresult.QueryResult;
import ru.yandex.partner.test.utils.TestUtils;

import static org.assertj.core.api.Assertions.assertThat;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@IntapiTest
public class PageOwnerReportTest {
    private static final String DATA_FILE_NAME = "ru.yandex.partner.intapi/antifrod/page_owner_report_test_data.json";

    @Autowired
    DbQueryService dbQueryService;

    @Autowired
    MockMvc mvc;

    @Autowired
    ACLConfig aclConfig;

    @Autowired
    IACLRepository aclRepository;

    /**
     * Метод читает тестовые данные типа PageOwnerData из файла в JSON формате
     *
     * @return Список объектов типа PageOwnerData
     * @throws IOException
     */
    static QueryResult<PageOwnerData> getReportFromResource() throws IOException {
        return IntApiTestUtils.readJsonFromResource(DATA_FILE_NAME, new TypeReference<QueryResult<PageOwnerData>>() {
        });
    }

    /**
     * Метод создаёт объект QueryResult на основе ответа URL
     *
     * @param url URL запроса
     * @return объект QueryResult
     * @throws Exception
     */
    QueryResult<PageOwnerData> getJSONReportFromURL(String url) throws Exception {
        MvcResult result = mvc.perform(get(url)).andExpect(status().isOk()).andReturn();

        return IntApiTestUtils.getMapper()
                .readValue(
                        result.getResponse().getContentAsString(),
                        new TypeReference<QueryResult<PageOwnerData>>() {
                        }
                );
    }

    @Test
    void checkDbValues() throws Exception {
        assertThat(dbQueryService.fetchAll(null))
                .containsExactlyInAnyOrder(
                        PageOwnerReportTest.getReportFromResource().getData().toArray(PageOwnerData[]::new)
                );
    }

    @ParameterizedTest
    @ValueSource(strings = {
            "/intapi/antifrod/page_owner_report",
            "/intapi/antifrod/page_owner_report.json",
            "/intapi/antifrod/page_owner_report?format=json",
    })
    void checkJsonResponse(String url) throws Exception {
        aclRepository.clearACLCache();
        IntApiTestUtils.addLocalAclDbRecord(dbQueryService);
        QueryResult<PageOwnerData> queryResult = this.getJSONReportFromURL(url);
        if (TestUtils.needSelfUpdate()) {
            TestUtils.getTestDataAndUpdateIfNeeded(queryResult, DATA_FILE_NAME);
            Assertions.fail("Self-updated " + DATA_FILE_NAME);
        }
        QueryResult<PageOwnerData> directJsonResult = PageOwnerReportTest.getReportFromResource();
        assertThat(queryResult.getOptions().getWithHeaders())
                .isTrue();
        assertThat(queryResult.getFields())
                .isEqualTo(directJsonResult.getFields());
        assertThat(queryResult.getData())
                .containsExactlyInAnyOrder(directJsonResult.getData().toArray(PageOwnerData[]::new));
    }

}
