package ru.yandex.partner.jsonapi.configuration;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.crnk.core.boot.CrnkBoot;
import io.crnk.core.boot.CrnkProperties;
import io.crnk.core.engine.properties.PropertiesProvider;
import io.crnk.core.engine.registry.ResourceRegistry;
import io.crnk.core.engine.url.ConstantServiceUrlProvider;
import io.crnk.core.module.ModuleRegistry;
import io.crnk.core.module.discovery.ServiceDiscovery;
import io.crnk.core.queryspec.mapper.DefaultQuerySpecUrlMapper;
import io.crnk.core.queryspec.mapper.QuerySpecUrlMapper;
import io.crnk.core.queryspec.pagingspec.OffsetLimitPagingBehavior;
import io.crnk.core.queryspec.pagingspec.OffsetLimitPagingSpec;
import io.crnk.core.queryspec.pagingspec.PagingBehavior;
import io.crnk.servlet.internal.ServletModule;
import io.crnk.spring.exception.SpringExceptionModule;
import io.crnk.spring.internal.SpringServiceDiscovery;
import io.crnk.spring.setup.boot.core.CrnkBootConfigurer;
import io.crnk.spring.setup.boot.core.CrnkCoreProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.jsonapi.crnk.DynamicModule;
import ru.yandex.partner.jsonapi.crnk.DynamicResourceRepository;
import ru.yandex.partner.jsonapi.models.ApiService;
import ru.yandex.partner.jsonapi.service.CrnkAdditionalDataService;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;


/**
 * Current crnk configuration with JSON API compliance, QuerySpec and module support.
 * Note that there is no support for QueryParams is this version due to the lack of JSON API compatibility.
 */
@Configuration
@ConditionalOnProperty(prefix = "crnk", name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(CrnkCoreProperties.class)
public class CrnkConfiguration implements ApplicationContextAware {

    private static final Long DEFAULT_PAGE_SIZE = 100L;
    private static final Long MAX_PAGE_SIZE = 1000L;

    private ApplicationContext applicationContext;

    private CrnkCoreProperties properties;

    private ObjectMapper objectMapper;

    @Autowired(required = false)
    private List<CrnkBootConfigurer> configurers;

    @Autowired
    public CrnkConfiguration(
            CrnkCoreProperties properties,
            ObjectMapper objectMapper
    ) {
        this.properties = properties;
        this.objectMapper = objectMapper;
    }

    @Bean
    public SpringServiceDiscovery discovery() {
        return new SpringServiceDiscovery();
    }

    @Bean
    public CrnkBoot crnkBoot(ServiceDiscovery serviceDiscovery) {
        CrnkBoot boot = new CrnkBoot();
        boot.setWebPathPrefix("/v1");
        boot.setObjectMapper(objectMapper);

        if (properties.getDomainName() != null && properties.getPathPrefix() != null) {
            String baseUrl = properties.getDomainName() + properties.getPathPrefix();
            boot.setServiceUrlProvider(new ConstantServiceUrlProvider(baseUrl));
        }
        boot.setServiceDiscovery(serviceDiscovery);
        boot.setPropertiesProvider(new PropertiesProvider() {
            @Override
            public String getProperty(String key) {
                if (CrnkProperties.RESOURCE_DEFAULT_DOMAIN.equals(key)) {
                    return properties.getDomainName();
                }
                if (CrnkProperties.ENFORCE_ID_NAME.equals(key)) {
                    return String.valueOf(properties.isEnforceIdName());
                }
                if (CrnkProperties.WEB_PATH_PREFIX.equals(key)) {
                    return properties.getPathPrefix();
                }
                if (CrnkProperties.ALLOW_UNKNOWN_ATTRIBUTES.equals(key)) {
                    return String.valueOf(properties.getAllowUnknownAttributes());
                }
                if (CrnkProperties.ALLOW_UNKNOWN_PARAMETERS.equals(key)) {
                    return String.valueOf(Optional.ofNullable(properties.getAllowUnknownParameters()).orElse(true));
                }
                if (CrnkProperties.RETURN_404_ON_NULL.equals(key)) {
                    return String.valueOf(properties.getReturn404OnNull());
                }
                return applicationContext.getEnvironment().getProperty(key);
            }
        });
        boot.addModule(new ServletModule(boot.getModuleRegistry().getHttpRequestContextProvider()));
        boot.addModule(new SpringExceptionModule());

        boot.setDefaultPageLimit(Optional.ofNullable(properties.getDefaultPageLimit()).orElse(DEFAULT_PAGE_SIZE));
        boot.setMaxPageLimit(Optional.ofNullable(properties.getMaxPageLimit()).orElse(MAX_PAGE_SIZE));

        if (configurers != null) {
            for (CrnkBootConfigurer configurer : configurers) {
                configurer.configure(boot);
            }
        }

        if (properties.getEnforceDotSeparator() != null) {
            QuerySpecUrlMapper urlMapper = boot.getUrlMapper();
            if (urlMapper instanceof DefaultQuerySpecUrlMapper) {
                ((DefaultQuerySpecUrlMapper) urlMapper).setEnforceDotPathSeparator(properties.getEnforceDotSeparator());
            }
        }

        boot.boot();
        return boot;
    }

    @Bean
    public QuerySpecUrlMapper querySpecUrlMapper() {
        return new DefaultQuerySpecUrlMapper();
    }

    @Bean
    public PagingBehavior<OffsetLimitPagingSpec> offsetLimitPagingBehavior() {
        return new OffsetLimitPagingBehavior();
    }

    @Bean
    public ResourceRegistry resourceRegistry(CrnkBoot boot) {
        return boot.getResourceRegistry();
    }

    @Bean
    public ModuleRegistry moduleRegistry(CrnkBoot boot) {
        return boot.getModuleRegistry();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    @Bean
    public <T extends ModelWithId> DynamicModule dynamicModule(List<ApiService<T>> apiServices,
                                                               AuthenticationFacade authenticationFacade,
                                                               CrnkAdditionalDataService crnkAdditionalDataService
    ) {
        return new DynamicModule(apiServices.stream()
                .map(apiService -> new DynamicResourceRepository<>(
                        apiService,
                        authenticationFacade,
                        objectMapper,
                        crnkAdditionalDataService))
                .collect(Collectors.toList())
        );
    }


}
