package ru.yandex.partner.jsonapi.configuration;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.web.authentication.AuthenticationFilter;
import org.springframework.security.web.context.SecurityContextPersistenceFilter;
import org.springframework.security.web.servletapi.SecurityContextHolderAwareRequestFilter;
import org.springframework.web.filter.RequestContextFilter;

import ru.yandex.partner.jsonapi.filter.action.ActionUserIdSettingFilter;
import ru.yandex.partner.jsonapi.filter.csrf.StatelessCsrfFilter;
import ru.yandex.partner.jsonapi.filter.limit.RequestLimitFilter;
import ru.yandex.partner.jsonapi.filter.locale.LocaleResolverFilter;
import ru.yandex.partner.jsonapi.filter.logging.LoggingFilter;
import ru.yandex.partner.libs.auth.filter.ExceptionFormatterFilter;

@Configuration
@Import(SecurityFilterConfiguration.class)
public class SecurityConfiguration extends WebSecurityConfigurerAdapter {

    private final AuthenticationFilter authenticationFilter;
    private final ExceptionFormatterFilter exceptionFormatterFilter;
    private final RequestLimitFilter requestLimitFilter;
    private final RequestContextFilter requestContextFilter;
    private final StatelessCsrfFilter statelessCsrfFilter;
    private final LoggingFilter loggingFilter;
    private final LocaleResolverFilter localeResolverFilter;
    private final ActionUserIdSettingFilter actionUserIdSettingFilter;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public SecurityConfiguration(AuthenticationFilter authenticationFilter,
                                 ExceptionFormatterFilter exceptionFormatterFilter,
                                 RequestLimitFilter requestLimitFilter,
                                 RequestContextFilter requestContextFilter,
                                 StatelessCsrfFilter statelessCsrfFilter,
                                 LoggingFilter loggingFilter,
                                 LocaleResolverFilter localeResolverFilter,
                                 ActionUserIdSettingFilter actionUserIdSettingFilter) {
        this.authenticationFilter = authenticationFilter;
        this.exceptionFormatterFilter = exceptionFormatterFilter;
        this.requestLimitFilter = requestLimitFilter;
        this.requestContextFilter = requestContextFilter;
        this.statelessCsrfFilter = statelessCsrfFilter;
        this.loggingFilter = loggingFilter;
        this.localeResolverFilter = localeResolverFilter;
        this.actionUserIdSettingFilter = actionUserIdSettingFilter;
    }

    @Override
    protected void configure(final HttpSecurity http) throws Exception {
        // Настройки фильтров
        http.sessionManagement()/* сессии не храним */
                .sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                .and()
                /* фильтр, который превращает ошибки в красивый ответ */
                .addFilterBefore(exceptionFormatterFilter, SecurityContextPersistenceFilter.class)
                /* фильтр, который выставляет контекст запроса */
                .addFilterBefore(requestContextFilter, ExceptionFormatterFilter.class)
                /* фильтр, который осущетвляет аутентификацию */
                .addFilterBefore(authenticationFilter, SecurityContextHolderAwareRequestFilter.class)
                /* фильтр добавляющий в контекст логера информацию о запросе */
                .addFilterAfter(loggingFilter, AuthenticationFilter.class)
                /* фильтр добавляющий в контекст локализации */
                .addFilterAfter(localeResolverFilter, LoggingFilter.class)
                /* фильтр, ограничивающий кол-во зпросов */
                .addFilterAfter(requestLimitFilter, LocaleResolverFilter.class)
                /* фильтр, проверяющий csrf токен */
                .addFilterAfter(statelessCsrfFilter, RequestLimitFilter.class)
                /* фильтр, который устанавливает userId для экшенов */
                .addFilterAfter(actionUserIdSettingFilter, StatelessCsrfFilter.class)
                // Общие настройки
                .csrf().disable()
                .formLogin().disable()
                .httpBasic().disable()
                .logout().disable()
                .anonymous().disable();
    }
}
