package ru.yandex.partner.jsonapi.configuration;

import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.web.authentication.AuthenticationFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.NegatedRequestMatcher;
import org.springframework.security.web.util.matcher.OrRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.web.filter.RequestContextFilter;
import org.springframework.web.servlet.LocaleResolver;

import ru.yandex.partner.core.action.ActionUserIdContext;
import ru.yandex.partner.core.configuration.BlackBoxAndTvmConfiguration;
import ru.yandex.partner.jsonapi.controller.AppVersionController;
import ru.yandex.partner.jsonapi.controller.RootController;
import ru.yandex.partner.jsonapi.filter.action.ActionUserIdSettingFilter;
import ru.yandex.partner.jsonapi.filter.limit.RequestLimitFilter;
import ru.yandex.partner.jsonapi.filter.locale.LocaleResolverFilter;
import ru.yandex.partner.jsonapi.filter.logging.LoggingFilter;
import ru.yandex.partner.jsonapi.response.ExceptionToErrorIdMapper;
import ru.yandex.partner.jsonapi.response.PartnerErrorResponseBuilder;
import ru.yandex.partner.jsonapi.response.PartnerExceptionResponseWriter;
import ru.yandex.partner.jsonapi.response.PartnerExceptionToErrorIdMapper;
import ru.yandex.partner.libs.auth.configuration.BasicSecurityConfiguration;
import ru.yandex.partner.libs.auth.converter.UserAuthenticationConverter;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.filter.ExceptionFormatterFilter;
import ru.yandex.partner.libs.auth.filter.ExceptionResponseWriter;
import ru.yandex.partner.libs.auth.handler.DoNothingAuthenticationSuccessHandler;
import ru.yandex.partner.libs.auth.handler.ExceptionAuthenticationFailureHandler;

@Configuration
@Import({BasicSecurityConfiguration.class,
        BlackBoxAndTvmConfiguration.class})
public class SecurityFilterConfiguration {

    @Bean
    public ExceptionToErrorIdMapper exceptionToErrorIdMapper() {
        return new PartnerExceptionToErrorIdMapper();
    }

    @Bean
    public ExceptionResponseWriter exceptionResponseWriter(
            ObjectMapper objectMapper,
            PartnerErrorResponseBuilder partnerErrorResponseBuilder
    ) {
        return new PartnerExceptionResponseWriter(objectMapper, partnerErrorResponseBuilder);
    }

    @Bean
    public RequestMatcher protectedRequestMatcher() {
        RequestMatcher publicRequestMatcher = new OrRequestMatcher(
                //add other public urls here
                new AntPathRequestMatcher("/actuator/**"),
                new AntPathRequestMatcher("/" + RootController.PATH + "/**"),
                new AntPathRequestMatcher("/" + AppVersionController.PATH + "/**")
        );
        return new NegatedRequestMatcher(publicRequestMatcher);
    }

    /**
     * Фильтр, выполняющий авторизацию. Отдаём ему:
     * 1) менеджер аутентификации, который производит аутентификацию
     * 2) конвертер, формирующий запрос к менеджеру из http-запроса
     * 3) matcher запросов, которые нужно аутентифицировать
     * 4) обработчик успешной аутентификации, который ничего не делает
     * 5) обработчик неуспешной аутентификации, который бросает ResponseStatusException
     *
     * @param authenticationManager менеджер
     * @return фильтр
     */
    @Bean
    public AuthenticationFilter authenticationFilter(
            AuthenticationManager authenticationManager,
            RequestMatcher protectedRequestMatcher
    ) {
        AuthenticationFilter authenticationFilter = new AuthenticationFilter(
                authenticationManager, new UserAuthenticationConverter()
        );
        authenticationFilter.setRequestMatcher(protectedRequestMatcher);
        authenticationFilter.setSuccessHandler(new DoNothingAuthenticationSuccessHandler());
        authenticationFilter.setFailureHandler(new ExceptionAuthenticationFailureHandler());
        return authenticationFilter;
    }

    /**
     * Форматирование ответа при ошибках специфично для jsonapi,
     * поэтому этот фильтр создаётся здесь.
     *
     * @param exceptionResponseWriter "писатель исключений"
     * @return фильтр
     */
    @Bean
    public ExceptionFormatterFilter exceptionFormatterFilter(
            ExceptionResponseWriter exceptionResponseWriter
    ) {
        return new ExceptionFormatterFilter(exceptionResponseWriter);
    }

    @Bean
    @ConfigurationProperties("api.config.request.limit.personal")
    public Map<String, Integer> perUserLimitMap() {
        return new HashMap<>();
    }

    @Bean
    public RequestLimitFilter requestLimitFilter(
            AuthenticationFacade authenticationFacade,
            Map<String, Integer> perUserLimitMap,
            @Value("${api.config.request.limit.default:-1}") int defaultLimit
    ) {
        return new RequestLimitFilter(authenticationFacade, defaultLimit, perUserLimitMap);
    }

    @Bean
    public RequestContextFilter requestContextFilter() {
        return new RequestContextFilter();
    }

    @Bean
    public LoggingFilter loggingFilter(AuthenticationFacade authenticationFacade) {
        return new LoggingFilter(authenticationFacade);
    }

    @Bean
    public LocaleResolverFilter localeResolverFilter(LocaleResolver localeResolver) {
        return new LocaleResolverFilter(localeResolver);
    }

    @Bean
    public ActionUserIdSettingFilter actionUserIdSettingFilter(ActionUserIdContext actionUserIdContext,
                                                               AuthenticationFacade authenticationFacade) {
        return new ActionUserIdSettingFilter(actionUserIdContext, authenticationFacade);
    }
}
