package ru.yandex.partner.jsonapi.configuration;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Locale;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.ContentNegotiationConfigurer;
import org.springframework.web.servlet.config.annotation.DelegatingWebMvcConfiguration;

import ru.yandex.direct.utils.json.LocalDateDeserializer;
import ru.yandex.direct.utils.json.LocalDateSerializer;
import ru.yandex.direct.utils.json.LocalDateTimeDeserializer;
import ru.yandex.direct.utils.json.LocalDateTimeSerializer;
import ru.yandex.partner.defaultconfiguration.GracefulShutdown;
import ru.yandex.partner.jsonapi.configuration.locale.PartnerLocaleResolver;

/**
 * Класс для дополнительной настройки WebMVC
 */
@Configuration
@Import({GracefulShutdown.class})
public class WebConfiguration extends DelegatingWebMvcConfiguration {

    @Value("${app.locale.supported:ru,en,tr}")
    List<Locale> supportedLocales;

    @Value("${app.locale.default:ru}")
    Locale defaultLocale;

    @Bean
    @NotNull
    @Override
    public LocaleResolver localeResolver() {
        PartnerLocaleResolver localeResolver = new PartnerLocaleResolver();
        localeResolver.setSupportedLocales(supportedLocales);
        localeResolver.setDefaultLocale(defaultLocale);
        return localeResolver;
    }

    @Override
    public void configureContentNegotiation(ContentNegotiationConfigurer configurer) {
        configurer
                .mediaType("json", MediaType.APPLICATION_JSON)
                .ignoreAcceptHeader(true)
                .defaultContentType(MediaType.APPLICATION_JSON);
    }

    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = StreamEx.of(converters)
                .findFirst(MappingJackson2HttpMessageConverter.class::isInstance)
                .map(MappingJackson2HttpMessageConverter.class::cast)
                .orElse(null);
        if (mappingJackson2HttpMessageConverter != null) {
            SimpleModule dateTimeModule = new SimpleModule("LocalDateTimeModule");
            dateTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer());
            dateTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());

            SimpleModule dateModule = new SimpleModule("LocalDateModule");
            dateModule.addSerializer(LocalDate.class, new LocalDateSerializer());
            dateModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());

            mappingJackson2HttpMessageConverter.getObjectMapper()
                    .registerModule(dateTimeModule)
                    .registerModule(dateModule)
                    .disable(DeserializationFeature.ACCEPT_FLOAT_AS_INT);
        }
    }
}
