package ru.yandex.partner.jsonapi.configuration.locale;

import java.util.List;
import java.util.Locale;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.i18n.AcceptHeaderLocaleResolver;

import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxUserInfo;

public class PartnerLocaleResolver extends AcceptHeaderLocaleResolver {

    private static final Logger LOGGER = LoggerFactory.getLogger(PartnerLocaleResolver.class);

    private static final String PARAM_NAME = "lang";

    private final AuthenticationFacade authenticationFacade = new AuthenticationFacade();

    @Nonnull
    @Override
    public Locale resolveLocale(HttpServletRequest request) {
        String queryParamValue = request.getParameter(PARAM_NAME);
        if (queryParamValue != null) {
            Locale queryLocale = resolve(queryParamValue);
            if (queryLocale != null) {
                return queryLocale;
            }
        }
        Optional<String> userLanguage = Optional.ofNullable(authenticationFacade.getUserAuthentication())
                .map(UserAuthentication::getBlackboxUserInfo).map(BlackboxUserInfo::getLanguage);
        if (userLanguage.isPresent()) {
            Locale userLocale = resolve(userLanguage.get());
            if (userLocale != null) {
                return userLocale;
            }
        }
        return super.resolveLocale(request);
    }

    protected Locale resolve(String localeString) {
        try {
            Locale locale = parseLocaleValue(localeString);
            List<Locale> supportedLocales = getSupportedLocales();
            if (supportedLocales.isEmpty() || supportedLocales.contains(locale)) {
                return locale;
            }
        } catch (IllegalArgumentException ex) {
            LOGGER.debug("Ignoring invalid locale value [" + localeString + "]: " + ex.getMessage());
        }
        return null;
    }

    @Nullable
    protected Locale parseLocaleValue(String localeValue) {
        return StringUtils.parseLocale(localeValue);
    }
}
