package ru.yandex.partner.jsonapi.controller

import com.fasterxml.jackson.databind.ObjectMapper
import io.crnk.core.engine.document.Document
import io.crnk.core.utils.Nullable
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.partner.jsonapi.controller.meta.UacSaasMetaInformation
import ru.yandex.partner.jsonapi.response.dto.UacSaasAppDto
import ru.yandex.partner.jsonapi.service.UacSaasService
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade
import ru.yandex.partner.libs.extservice.direct.AppInfo
import ru.yandex.partner.libs.extservice.direct.CacheableDirectIntapiService

@RestController
class UacController
@Autowired
constructor(
    private val service: UacSaasService,
    private val objectMapper: ObjectMapper,
    private val authenticationFacade: AuthenticationFacade,
    private val directIntapiService: CacheableDirectIntapiService
) {
    private val logger = LoggerFactory.getLogger(UacController::class.java);

    /**
     * Ручка для хождения в saas-searchproxy
     *
     * @param text        - фильтр по префиксу
     * @param queryFields - поля по которым искать - по умолчанию поиск идет по z_title, s_bundle_id оператор ИЛИ
     * примеры другие полей: s_platform, z_author, s_app_id
     * @param pageNum     - номер страницы
     * @param limit       - лимит вывода по умолчанию 10
     * @param store       - магазин приложений google_play, itunes
     * @param plainQuery  - использовать язык запросов в saas
     * @return возвращает список найденных приложений и количество найденных приложений
     */
    @GetMapping("/v1/api/uac/app_info/search")
    fun getUacSaas(
        @RequestParam text: String,
        @RequestParam(required = false) queryFields: List<String>?,
        @RequestParam(required = false) store: String?,
        @RequestParam(required = false) pageNum: Int?,
        @RequestParam(required = false, defaultValue = "10") limit: Int,
        @RequestParam(required = false, defaultValue = "false") plainQuery: Boolean
    ): Document {
        val (meta, apps) = service.getData(
            text = text,
            queryFields = queryFields,
            store = store,
            pageNum = pageNum,
            limit = limit,
            plainQuery = plainQuery,
            lang = authenticationFacade.lang
        )

        return makeResponse(meta, apps)
    }

    @PostMapping("/v1/api/uac/app_info/add")
    fun appInfoParsing(
        @RequestParam url: String
    ): Document {
        val appInfo = directIntapiService.appInfoParsing(url)
        return makeResponse(appInfo)
    }

    @PostMapping("/v1/api/uac/app_info/search_or_add")
    fun appInfo(
        @RequestParam text: String,
        @RequestParam(required = false) store: String?,
        @RequestParam(required = false) pageNum: Int?,
        @RequestParam(required = false, defaultValue = "10") limit: Int,
    ): Document {

        if (isUrl(text)) {
            logger.info("try add by direct:  {}", text)
            // нет в suggest пробуем прокачать
            val appInfo = directIntapiService.appInfoParsing(text)

            return makeResponse(appInfo)
        } else {
            logger.info("try search by saas:  {}", text)
            val (meta, apps) = service.getData(
                text = text,
                queryFields = null,
                store = store,
                pageNum = pageNum,
                limit = limit,
                plainQuery = false,
                lang = authenticationFacade.lang
            )
            // Это не ссылка или не нашлось и в direct
            return makeResponse(meta, apps)
        }
    }

    private fun makeResponse(meta: UacSaasMetaInformation, apps: List<UacSaasAppDto>): Document {
        val response = Document()

        response.data = Nullable.of(apps)
        response.meta = objectMapper.valueToTree(meta)

        return response
    }

    private fun makeResponse(appInfo: AppInfo?): Document {
        val response = Document()

        if (appInfo == null) {
            response.meta = objectMapper.valueToTree(UacSaasMetaInformation(0, false, 0))
            response.data = Nullable.empty()
        } else {
            response.meta = objectMapper.valueToTree(UacSaasMetaInformation(1, false, 1))
            response.data = Nullable.of(
                listOf(
                    mapOf<String, Any>(
                        "title" to appInfo.title,
                        "icon" to appInfo.iconUrl,
                        "platform" to platform(appInfo.platform),
                        "bundle" to appInfo.bundleId,
                        "author" to appInfo.vendor,
                        "region" to appInfo.region,
                        "url" to appInfo.url,
                        "storeId" to appInfo.appId
                    )
                )
            )
        }

        return response
    }

    private fun isUrl(text: String): Boolean {
        val regex = "^http(s)?://.+".toRegex()
        return regex.matches(text)
    }

    fun platform(intPlatform: Int): String {
        return when (intPlatform) {
            1 -> "android"
            2 -> "ios"
            else -> "unknown"
        }
    }
}
