package ru.yandex.partner.jsonapi.crnk;

import java.io.IOException;
import java.util.Set;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import io.crnk.core.boot.CrnkBoot;
import io.crnk.servlet.CrnkFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.utils.ApiUtils;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;

@Component
public class CustomCrnkFilter extends CrnkFilter {
    private static final Logger LOGGER = LoggerFactory.getLogger(CustomCrnkFilter.class);

    /**
     * Путь обязательно должен иметь слеш на конце "/"
     */
    private final Set<String> skipServletPaths;

    private final RequestMatcher protectedRequestMatcher;

    @Autowired
    public CustomCrnkFilter(
            CrnkBoot boot,
            RequestMatcher protectedRequestMatcher,
            RequestMappingHandlerMapping mapping
    ) {
        super(boot);
        this.protectedRequestMatcher = protectedRequestMatcher;
        this.skipServletPaths = ApiUtils.getSpringEndpoints(mapping);
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException,
            ServletException {
        if (req instanceof HttpServletRequest && !skip((HttpServletRequest) req)) {
            if (protectedRequestMatcher.matches((HttpServletRequest) req)) {
                validateHeaders((HttpServletRequest) req);
            }
            if (!isActionRequest((HttpServletRequest) req)) {
                super.doFilter(req, res, chain);
                return;
            }
        }

        chain.doFilter(req, res);
    }

    private boolean isAddFieldRequest(HttpServletRequest request) {
        return "GET".equals(request.getMethod()) && request.getServletPath().contains("/add_fields");
    }

    private boolean isDependsRequest(HttpServletRequest request) {
        return "GET".equals(request.getMethod()) && request.getServletPath().contains("/depends");
    }

    private boolean isDefaultsRequest(HttpServletRequest request) {
        return "GET".equals(request.getMethod()) && request.getServletPath().contains("/defaults");
    }

    private boolean skip(HttpServletRequest request) {
        String originalServletPath = request.getServletPath();
        String checkableServletPath = originalServletPath.endsWith("/")
                ? originalServletPath
                : originalServletPath + "/";
        return skipServletPaths.contains(checkableServletPath.toLowerCase()) ||
                isAddFieldRequest(request) ||
                isDependsRequest(request) ||
                isDefaultsRequest(request);
    }

    private boolean isActionRequest(HttpServletRequest request) {
        return "POST".equals(request.getMethod()) && request.getServletPath().contains("/action/");
    }

    private void validateHeaders(HttpServletRequest request) {
        final String apiProtocol = "application/vnd.api+json";
        LOGGER.debug("Request Accept header is {}", request.getHeader("Accept"));
        if (!apiProtocol.equals(request.getHeader("Accept"))) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_ACCEPTABLE);
        }

        if (request.getContentLength() > -1 && !apiProtocol.equals(request.getContentType())) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__MEDIA_TYPE);
        }
    }
}
