package ru.yandex.partner.jsonapi.crnk;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import io.crnk.core.engine.document.ErrorDataBuilder;
import io.crnk.core.engine.information.resource.ResourceInformation;
import io.crnk.core.engine.query.QueryContext;
import io.crnk.core.exception.BadRequestException;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.queryspec.mapper.DefaultQuerySpecUrlMapper;
import io.crnk.core.queryspec.mapper.QueryParameter;
import io.crnk.core.queryspec.mapper.QueryParameterType;
import io.crnk.core.queryspec.mapper.QueryPathResolver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilterAdapter;
import ru.yandex.partner.jsonapi.crnk.filter.parser.CrnkFilterParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.FilterNode;
import ru.yandex.partner.jsonapi.messages.JsonApiModelMsg;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.jsonapi.response.ExceptionToErrorIdMapper;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

/**
 * Этот класс отвечает за то, чтобы десериализовать query-параметры в объект QuerySpec,
 * а также за то, чтобы сериализовать такой обект обратно при построении ссылок
 * <p>
 * Объект QuerySpec не гибкий и плохо поддаётся переопределению,
 * поэтому мы кладём в него "фильтр" {@link CustomParametersHolder}, в который помещаем дополнительные
 * сведения о параметрах, в том числе оригинальные параметры запроса.
 * <p>
 * Для работы с кастомными "фильтрами" используется класс {@link QuerySpecUtil}
 */
@Component
public class CustomQuerySpecUrlMapper extends DefaultQuerySpecUrlMapper implements MessageSourceAware {

    public static final String META_PARAM_NAME = "meta";
    public static final String META_TOTAL_PARAM_VALUE = "total";

    private final ExceptionToErrorIdMapper exceptionToErrorIdMapper;
    private final CrnkFilterParser crnkFilterParser;

    private MessageSourceAccessor messageSourceAccessor = new MessageSourceAccessor(new DelegatingMessageSource());

    @Autowired
    public CustomQuerySpecUrlMapper(ExceptionToErrorIdMapper exceptionToErrorIdMapper,
                                    CrnkFilterParser crnkFilterParser) {
        this.exceptionToErrorIdMapper = exceptionToErrorIdMapper;
        this.crnkFilterParser = crnkFilterParser;
        this.pathResolver = new CustomQueryPathResolver();
    }

    @Override
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        messageSourceAccessor = new MessageSourceAccessor(messageSource);
    }

    /**
     * Корневой метод десериализвции. Здесь добавляем оригинальные параметры
     */
    @Override
    public QuerySpec deserialize(ResourceInformation resourceInformation, Map<String, Set<String>> parameterMap,
                                 QueryContext queryContext) {
        QuerySpec querySpec = super.deserialize(resourceInformation, parameterMap, queryContext);
        QuerySpecUtil.setOriginalParameters(querySpec, parameterMap);
        return querySpec;
    }

    /**
     * Десериализация фильтров у нас кастомная, нас интересует только параметр "filter" без квадратных скобок
     */
    @Override
    protected void deserializeFilter(QuerySpec querySpec, QueryParameter parameter, QueryContext queryContext) {
        if (QueryParameterType.FILTER.name().equalsIgnoreCase(parameter.getName())
                && !parameter.getValues().isEmpty()) {
            String filterString = parameter.getValues().iterator().next();

            if (!filterString.isEmpty()) {
                FilterNode filterNode = crnkFilterParser.parse(filterString);
                querySpec.addFilter(new CrnkFilterAdapter(filterString, filterNode));
            }
        }
    }

    @Override
    protected void deserializeFields(QuerySpec querySpec, QueryParameter parameter, QueryContext queryContext) {
        try {
            super.deserializeFields(querySpec, parameter, queryContext);
        } catch (BadRequestException e) {
            List<String> errors = getUnexpectedFields(parameter, queryContext);
            ErrorDataBuilder errorDataBuilder = new ErrorDataBuilder()
                    .setId("" + exceptionToErrorIdMapper.mapHttpStatusToErrorId(e.getHttpStatus()))
                    .setTitle(messageSourceAccessor.getMessage(JsonapiErrorMsg.BAD_REQUEST))
                    .setDetail(messageSourceAccessor.getMessage(MsgWithArgs.of(JsonApiModelMsg.UNEXPECTED_FIELDS,
                            String.join(", ", errors))));

            throw new BadRequestException(HttpStatus.BAD_REQUEST.value(), errorDataBuilder.build(), e);
        }
    }


    /**
     * Сюда приходят кастомные параметры, например meta=total,
     * флаг о его присутствии ставится в {@link CustomParametersHolder}
     */
    @Override
    protected void deserializeUnknown(QuerySpec querySpec, QueryParameter parameter) {
        if (META_PARAM_NAME.equalsIgnoreCase(parameter.getName()) && !parameter.getValues().isEmpty()
                && parameter.getValues().stream().allMatch(META_TOTAL_PARAM_VALUE::equalsIgnoreCase)) {
            QuerySpecUtil.setTotalRequested(querySpec);
        } else {
            super.deserializeUnknown(querySpec, parameter);
        }
    }

    /**
     * Корень сериализации.
     * Прежде чем писать параметры, достаём оригинальные, чтобы не потерять неучтённые, напимер lang
     */
    @Override
    protected void serialize(QuerySpec querySpec, Map<String, Set<String>> map, QuerySpec rootQuerySpec,
                             QueryContext queryContext) {
        map.putAll(QuerySpecUtil.getOriginalParameters(querySpec));
        map.putAll(QuerySpecUtil.getPageBehaviorParameters(querySpec));

        super.serialize(querySpec, map, rootQuerySpec, queryContext);
        map.remove("include");
    }

    /**
     * Здесь сериализуем то, что лежит в поле "фильтры" {@link QuerySpec} - наш кастомный фильтр
     * и {@link CustomParametersHolder}
     */
    @Override
    protected void serializeFilters(QuerySpec querySpec, ResourceInformation resourceInformation,
                                    Map<String, Set<String>> map, boolean isRoot, QueryContext queryContext) {
        QuerySpecUtil.getCrnkFilterAdapter(querySpec).ifPresent(
                filterAdapter -> map.put(QueryParameterType.FILTER.name().toLowerCase(),
                        Set.of(filterAdapter.getFilterString()))
        );

        if (QuerySpecUtil.isTotalRequested(querySpec)) {
            map.put(META_PARAM_NAME, Set.of(META_TOTAL_PARAM_VALUE));
        }
    }


    /**
     * У нас по умолчанию поле fields всегда с указанием ресурса в квадратных скобках,
     * поэтому здесь форсируем такое поведение
     */
    @Override
    protected String addResourceType(QueryParameterType type, String key, ResourceInformation resourceInformation,
                                     boolean isRoot) {
        return super.addResourceType(type, key, resourceInformation, isRoot && !QueryParameterType.FIELDS.equals(type));
    }

    private List<String> getUnexpectedFields(QueryParameter parameter, QueryContext queryContext) {
        ResourceInformation resourceInformation = parameter.getResourceInformation();
        List<String> unexpectedFields = new ArrayList<>();
        for (String values : parameter.getValues()) {
            for (String value : splitValues(values)) {
                if (isUnexpectedField(parameter, resourceInformation, value, queryContext)) {
                    unexpectedFields.add(value);
                }
            }
        }

        return unexpectedFields;
    }

    private boolean isUnexpectedField(QueryParameter parameter, ResourceInformation resourceInformation, String value,
                                      QueryContext queryContext) {
        try {
            List<String> attributePath = splitAttributePath(value, parameter);
            pathResolver.resolve(resourceInformation, attributePath, QueryPathResolver.NamingType.JSON,
                    parameter.getName(), queryContext);
            return false;
        } catch (BadRequestException e) {
            return true;
        }
    }

    private String[] splitValues(String values) {
        return values.split(",");
    }

}
