package ru.yandex.partner.jsonapi.crnk;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import io.crnk.core.engine.document.Resource;
import io.crnk.core.engine.http.HttpRequestContextAware;
import io.crnk.core.engine.http.HttpRequestContextProvider;
import io.crnk.core.engine.information.InformationBuilder;
import io.crnk.core.engine.information.resource.ResourceFieldType;
import io.crnk.core.engine.registry.RegistryEntryBuilder;
import io.crnk.core.module.InitializingModule;
import io.crnk.core.queryspec.pagingspec.NumberSizePagingSpec;

import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.models.relationships.ApiRelationship;
import ru.yandex.partner.jsonapi.models.relationships.RelationshipRepositoryImpl;

public class DynamicModule implements InitializingModule, HttpRequestContextAware {
    private final Map<String, DynamicResourceRepository<?>> dynamicResourceRepositories;
    private ModuleContext context;
    private HttpRequestContextProvider requestContextProvider;

    public DynamicModule(List<DynamicResourceRepository<?>> dynamicResourceRepositories) {
        this.dynamicResourceRepositories = dynamicResourceRepositories.stream()
                .collect(Collectors.toMap(DynamicResourceRepository::getResourceType, Function.identity()));
    }

    @Override
    public void init() {
        for (DynamicResourceRepository<?> dynamicResourceRepository : dynamicResourceRepositories.values()) {
            RegistryEntryBuilder builder = context.newRegistryEntryBuilder();
            dynamicResourceRepository.setHttpRequestContextProvider(requestContextProvider);

            RegistryEntryBuilder.ResourceRepositoryEntryBuilder resourceRepository = builder.resourceRepository();
            resourceRepository.instance(dynamicResourceRepository);

            InformationBuilder.ResourceInformationBuilder resource = builder.resource();
            resource.resourceType(dynamicResourceRepository.getResourceType());
            resource.implementationType(Resource.class);
            resource.pagingSpecType(NumberSizePagingSpec.class);

            addFields(dynamicResourceRepository, resource);
            for (ApiRelationship<?> relationship : dynamicResourceRepository.getApiRelationships()) {
                var relationshipDynamicRepository =
                        dynamicResourceRepositories.get(relationship.getApiResourceType().getResourceType());
                if (relationshipDynamicRepository == null) {
                    throw new IllegalStateException("There is no DynamicResourceRepository for the resource '"
                            + relationship.getApiResourceType()
                            + "'. Error while creating DynamicResourceRepository for the resource '"
                            + dynamicResourceRepository.getResourceType() + "'");
                }

                RegistryEntryBuilder.RelationshipRepositoryEntryBuilder parentRepository =
                        builder.relationshipRepositoryForField(relationship.getField());

                // даже если связь один-к-одному все равно используем many (исторически из перла)
                parentRepository.instance(new RelationshipRepositoryImpl<>(relationship,
                        relationshipDynamicRepository));

                Class<?> clazz = relationship.isOne()
                        ? Resource.class
                        : List.class;
                resource.addField(relationship.getField(), ResourceFieldType.RELATIONSHIP, clazz)
                        .oppositeResourceType(relationship.getApiResourceType().getResourceType());
            }

            context.addRegistryEntry(builder.build());
        }
    }

    private void addFields(DynamicResourceRepository<?> dynamicResourceRepository,
                           InformationBuilder.ResourceInformationBuilder resource) {
        var apiFields = dynamicResourceRepository.getApiFields();
        var resourceIds = new ArrayList<>(apiFields.size());
        for (ApiField<?> field : apiFields) {
            Class<?> fieldClass;
            if (ResourceFieldType.ID == field.getResourceFieldType()) {
                /*
                 * Если указать другой класс падает на парсинге
                 */
                fieldClass = String.class;
                resourceIds.add(field.getJsonName());
            } else {
                fieldClass = field.getValueClass().getRawClass();
            }

            resource.addField(field.getJsonName(), field.getResourceFieldType(), fieldClass);
        }

        if (resourceIds.size() != 1) {
            throw new IllegalStateException("ResourceType.ID expected count: 1, actual count: " +
                    resourceIds.size());
        }
    }

    @Override
    public String getModuleName() {
        return "dynamic";
    }

    @Override
    public void setupModule(ModuleContext context) {
        this.context = context;
    }

    @Override
    public void setHttpRequestContextProvider(HttpRequestContextProvider requestContextProvider) {
        this.requestContextProvider = requestContextProvider;
    }

    @Nullable
    public DynamicResourceRepository<?> getDynamicResourceRepository(String resourceType) {
        return dynamicResourceRepositories.get(resourceType);
    }
}

