package ru.yandex.partner.jsonapi.crnk;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonFormat.Shape;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.crnk.core.engine.document.Document;
import io.crnk.core.engine.document.Resource;
import io.crnk.core.engine.http.HttpRequestContext;
import io.crnk.core.engine.http.HttpRequestContextProvider;
import io.crnk.core.engine.result.Result;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.queryspec.SortSpec;
import io.crnk.core.repository.LinksRepository;
import io.crnk.core.repository.MetaRepository;
import io.crnk.core.repository.ResourceRepositoryBase;
import io.crnk.core.repository.UntypedResourceRepository;
import io.crnk.core.resource.links.LinksInformation;
import io.crnk.core.resource.list.DefaultResourceList;
import io.crnk.core.resource.list.ResourceList;
import io.crnk.core.resource.meta.MetaInformation;
import io.crnk.core.utils.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.util.CollectionUtils;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.partner.core.action.ActionPayloadType;
import ru.yandex.partner.core.action.factories.FieldSetType;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.core.utils.OrderBy.Direction;
import ru.yandex.partner.jsonapi.constants.CrnkConstants;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.crnk.fields.ModelPathForApi;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilterAdapter;
import ru.yandex.partner.jsonapi.crnk.filter.parser.FilterNode;
import ru.yandex.partner.jsonapi.messages.CommonMsg;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.jsonapi.models.ApiService;
import ru.yandex.partner.jsonapi.models.relationships.ApiRelationship;
import ru.yandex.partner.jsonapi.service.CrnkAdditionalDataService;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

import static ru.yandex.partner.core.utils.OrderBy.Direction.ASC;
import static ru.yandex.partner.core.utils.OrderBy.Direction.DESC;
import static ru.yandex.partner.jsonapi.utils.ApiUtils.parseJsonMap;

/*
 * Нельзя делать компонентом спринга (падает при старте) и поэтому Интерфейс HttpRequestContextAware не работает
 */
public class DynamicResourceRepository<M extends ModelWithId>
        extends ResourceRepositoryBase<Resource, String>
        implements UntypedResourceRepository<Resource, String>, MetaRepository<Resource>, LinksRepository<Resource> {

    private static final Logger LOGGER = LoggerFactory.getLogger(DynamicResourceRepository.class);
    private static final EnumSet<HttpMethod> SELF_LINKING_METHODS = EnumSet.of(HttpMethod.POST, HttpMethod.PATCH);
    private static final String HTTP_POST = HttpMethod.POST.name();
    private static final String HTTP_PATCH = HttpMethod.PATCH.name();

    private final ApiService<M> apiService;
    private final ApiModel<M> apiModel;
    private final ObjectMapper objectMapper;
    private final Map<String, ApiField<M>> apiFieldMap;
    private final AuthenticationFacade authenticationFacade;
    private final boolean hasActionField;
    private final Set<String> actionsReturnFields;
    private HttpRequestContextProvider httpRequestContextProvider;
    private final CrnkAdditionalDataService crnkAdditionalDataService;

    public DynamicResourceRepository(ApiService<M> apiService,
                                     AuthenticationFacade authenticationFacade,
                                     ObjectMapper objectMapper,
                                     CrnkAdditionalDataService crnkAdditionalDataService) {
        super(Resource.class);

        this.apiService = apiService;
        this.authenticationFacade = authenticationFacade;
        this.apiModel = apiService.getApiModel();
        this.objectMapper = objectMapper;
        this.crnkAdditionalDataService = crnkAdditionalDataService;

        this.apiFieldMap = apiModel.getApiFieldMap();

        // хорошо ли зашивать имя, если в ApiActionsModelPart есть возможность менять???
        this.hasActionField = apiFieldMap.containsKey(CrnkConstants.ACTIONS_FIELD_NAME);
        var set = Set.of(CrnkConstants.ACTIONS_FIELD_NAME,
                CrnkConstants.MULTISTATE_FIELD_NAME,
                CrnkConstants.MULTISTATE_NAME_FIELD_NAME);
        this.actionsReturnFields =
                apiService.getApiModel().getFields().stream().map(ApiField::getJsonName)
                        .filter(set::contains).collect(Collectors.toSet());
    }

    public void setHttpRequestContextProvider(HttpRequestContextProvider requestContextProvider) {
        this.httpRequestContextProvider = requestContextProvider;
    }

    public HttpRequestContextProvider getHttpRequestContextProvider() {
        return httpRequestContextProvider;
    }

    @Override
    public String getResourceType() {
        return apiModel.getResourceType();
    }

    public CrnkAdditionalDataService getCrnkAdditionalDataService() {
        return crnkAdditionalDataService;
    }

    public List<ApiField<M>> getApiFields() {
        return apiService.getApiFieldsService().getPermittedApiFields();
    }

    @Override
    public Resource findOne(String id, QuerySpec querySpec) {
        if (!"GET".equals(httpRequestContextProvider.getRequestContext().getMethod())) {
            // не интересен поиск при PATCH отдаем как будто бы что-то нашли,
            // метод save сам разберется с поиском элемента
            var resource = new Resource();
            resource.setId(id);
            resource.setType(querySpec.getResourceType());
            return resource;
        }

        Set<String> apiRequestedFields = crnkAdditionalDataService.getRequestedFieldsWithModelDefaults(querySpec);

        List<ApiRelationship<M>> apiRelationships = crnkAdditionalDataService.getApiRelationships(querySpec,
                apiService);
        Set<String> apiIncludedFields = crnkAdditionalDataService.getApiIncludedFields(apiRelationships);
        Set<String> requestFields = Sets.union(apiRequestedFields, apiIncludedFields);

        Set<ModelProperty<? extends Model, ?>> requestProperties =
                apiService.getApiFieldsService().modelPropertiesFromRequestedFieldNames(requestFields);

        M model = apiService.findOne(apiIdToCoreId(id), requestProperties);

        if (model == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }

        return convert(model, apiRequestedFields, apiRelationships);
    }

    @Override
    public ResourceList<Resource> findAll(QuerySpec querySpec) {
        FilterNode filterNode = getFilterNode(querySpec);

        List<OrderBy> orderByList = getOrderByListForQuerySpec(querySpec).orElse(null);

        LimitOffset limitOffset = QuerySpecUtil.getLimitOffset(querySpec).orElse(null);

        Set<String> apiRequestedFields = crnkAdditionalDataService.getRequestedFieldsWithModelDefaults(querySpec);
        List<ApiRelationship<M>> apiRelationships = crnkAdditionalDataService.getApiRelationships(querySpec,
                apiService);
        Set<String> apiIncludedFields = crnkAdditionalDataService.getApiIncludedFields(apiRelationships);
        Set<String> requestFields = Sets.union(apiRequestedFields, apiIncludedFields);

        Set<ModelProperty<? extends Model, ?>> requestProperties =
                apiService.getApiFieldsService().modelPropertiesFromRequestedFieldNames(requestFields);

        List<M> list = apiService.findAll(requestProperties, filterNode, limitOffset, orderByList);

        HttpRequestContext httpRequestContext = httpRequestContextProvider.getRequestContext();

        return new DefaultResourceList<>(list.stream()
                .map(model -> convert(model, apiRequestedFields, apiRelationships))
                .collect(Collectors.toList()),
                crnkAdditionalDataService.getBodyMetaInformation(querySpec, filterNode, list.size(), apiService),
                crnkAdditionalDataService.getBodyLinksInformation(querySpec, httpRequestContext, apiService)
        );
    }

    @Override
    public <S extends Resource> S save(S resource) {
        var savedModel = persist(resource, (parsedModel, updatedModelProperties, errorParsing) -> {
            var id = Objects.requireNonNull(resource.getId());
            parsedModel.setId(apiIdToCoreId(id));
            return apiService.save(parsedModel, updatedModelProperties, errorParsing);
        });

        Resource savedResource = convert(savedModel, resource.getAttributes().keySet());
        // TODO PERL TEST SPECIFICS (see one_object =>) uncomment and regenerate test
        setSelfLink(savedResource);

        //noinspection unchecked (у Resource нет наследников)
        return (S) savedResource;
    }

    @Override
    public <S extends Resource> S create(S resource) {
        var savedModel = persist(resource, apiService::create);

        Resource savedResource = convert(savedModel, Set.of());
        setSelfLink(savedResource);

        //noinspection unchecked (у Resource нет наследников)
        return (S) savedResource;
    }

    /**
     * В этом методе нельзя пользоваться некоторыми фишками, так как он попадает сюда не через
     * CrnkFilter, а через {@link ru.yandex.partner.jsonapi.controller.ActionsController}
     * <p>
     * Например, объект httpRequestContextProvider не будет заполнен
     */
    public Document doAction(String id, String actionName, String body, HttpRequestContext context) {
        if (!hasActionField) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }

        var factory = apiService.getActionFactory(actionName);
        if (factory == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }

        M updatedModel;
        Set<String> fieldsToReturn;

        var linksInformation = new BaseCrnkLinksInformation();

        if (factory.getType() == ActionPayloadType.MODEL) {
            Map<String, JsonNode> attributes;
            try {
                attributes = objectMapper.readValue(body, new TypeReference<>() {
                });
            } catch (JsonProcessingException e) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS, CommonMsg.INCORRECT_JSON, e);
            }

            var resource = new Resource();
            resource.setId(id);
            resource.setAttributes(attributes);
            var updatedFields = new IncomingApiFields<>(
                    Maps.<String, ApiField<M>>newHashMapWithExpectedSize(resource.getAttributes().size())
            );
            var errorParsing = new LinkedList<DefectInfo<Defect>>();

            var requestModel = parseModel(resource, updatedFields, errorParsing);
            requestModel.setId(apiIdToCoreId(id));

            updatedModel = apiService.doAction(actionName, requestModel, updatedFields, errorParsing);
            // TODO возможно экшен должен вернуть изменённые поля
            fieldsToReturn = updatedFields.keySet();
            linksInformation.setAddField(crnkAdditionalDataService.addFieldsLink(getResourceType(), context,
                    apiService));
        } else {
            var requestModel = apiService.getApiModel().createNewInstance();
            requestModel.setId(apiIdToCoreId(id));

            try {
                String payload = StringUtils.isEmpty(body) ? "{}" : body;
                updatedModel = apiService.doAction(actionName, requestModel, objectMapper.readTree(payload));
            } catch (JsonProcessingException e) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS, CommonMsg.INCORRECT_JSON, e);
            }

            if (factory.getReturnFields() == FieldSetType.DEFAULT) {
                fieldsToReturn = actionsReturnFields;
            } else {
                fieldsToReturn = Set.of();
            }
        }

        String selfLink = context.getBaseUrl() +
                context.getPath()
                        .substring(0, context.getPath().lastIndexOf(getResourceType()) + getResourceType().length())
                + "/" + coreIdToApiId(updatedModel.getId());
        var resource = convert(updatedModel, fieldsToReturn, selfLink);

        var document = new Document();
        document.setData(Nullable.of(resource));
        linksInformation.setSelf(selfLink);
        document.setLinks(objectMapper.valueToTree(linksInformation));

        var metaInformation = new BaseCrnkMetaInformation();
        metaInformation.setFields(fieldsToReturn);
        document.setMeta(objectMapper.valueToTree(metaInformation));
        return document;
    }

    public List<ApiRelationship<M>> getApiRelationships() {
        return apiService.getApiRelationships();
    }

    private <S extends Resource> M persist(S resource, Persister<M> persister) {
        var incomingApiFields =
                new IncomingApiFields<>(
                        Maps.<String, ApiField<M>>newHashMapWithExpectedSize(resource.getAttributes().size())
                );

        var errorParsing = new LinkedList<DefectInfo<Defect>>();

        M parsedModel = parseModel(resource, incomingApiFields, errorParsing);

        return persister.persist(parsedModel, incomingApiFields, errorParsing);
    }

    public <S extends Resource> M parseModel(S resource, IncomingApiFields<M> incomingApiFields,
                                             LinkedList<DefectInfo<Defect>> errorParsing) {
        var model = apiModel.createNewInstance();

        try {
            incomingApiFields.addIncomingFields(
                    parseJsonMap(model, resource.getAttributes(), apiFieldMap, incomingApiFields, errorParsing)
            );
        } catch (IOException e) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS);
        }

        return model;
    }

    private Resource convert(M model,
                             Set<String> requestedFields,
                             List<ApiRelationship<M>> apiRelationships) {
        var resource = convert(model, requestedFields);
        // TODO PERL TEST SPECIFICS (see one_object =>) uncomment and regenerate test
        setSelfLink(resource);
        for (ApiRelationship<M> apiRelationship : apiRelationships) {
            var relationship = apiRelationship.getRelationship(model);

            relationship.setLinks(objectMapper.valueToTree(crnkAdditionalDataService.relationshipLinksInformation(
                    resource, httpRequestContextProvider.getRequestContext(), apiRelationship)));
            resource.getRelationships().put(apiRelationship.getField(), relationship);
        }

        return resource;
    }

    private Resource convert(M model, Set<String> requestedFields) {
        var requestContext = httpRequestContextProvider.getRequestContext();
        return convert(model, requestedFields, requestContext.getBaseUrl() + requestContext.getPath());
    }

    private Resource convert(M model, Set<String> requestedFields, String url) {
        if (model == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }
        Resource resource = new Resource();
        resource.setId(coreIdToApiId(model.getId()));
        resource.setType(apiModel.getResourceType());

        var attributes = apiService.getApiFieldsService()
                .convertToApi(model,
                        requestedFields,
                        objectMapper,
                        authenticationFacade
                );


        resource.setAttributes(attributes);
        resource.setRelationships(crnkAdditionalDataService.getRelationships(resource.getId(), url, apiModel));

        if (attributes.isEmpty()) {
            // Чтобы в ответе сериализовывался пустой attributes
            return new PatchedResource(resource);
        } else {
            return resource;
        }
    }

    private String coreIdToApiId(Long id) {
        return apiService.coreIdToApiId(id);

    }

    private Long apiIdToCoreId(String id) {
        try {
            return apiService.apiIdToCoreId(id);
        } catch (Exception e) {
            LOGGER.debug("Error while convert Id from String to Long. Id = {}", id, e);
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__CONFLICT,
                    MsgWithArgs.of(CommonMsg.INCORRECT_ID, id));
        }
    }

    private Optional<List<OrderBy>> getOrderByListForQuerySpec(QuerySpec querySpec) {
        return Optional.of(querySpec.getSort())
                .map(l -> l.stream()
                        .map(this::sortSpecToOrderBy)
                        .filter(Optional::isPresent)
                        .map(Optional::get)
                        .collect(Collectors.toList())
                ).filter(l -> !l.isEmpty());
    }

    private Optional<OrderBy> sortSpecToOrderBy(SortSpec sortSpec) {
        return Optional.ofNullable(sortSpec.getAttributePath())
                .filter(l -> !CollectionUtils.isEmpty(l))
                // для полей корневой сущности длина массива всегда 1, для вложенных имя поля - последний элемент
                // (по вложенным сущностям пока не сортируем на самом деле)
                .map(l -> l.get(l.size() - 1))
                .map(apiFieldMap::get)
                .map(ApiField::getModelPath)
                .map(ModelPathForApi::rootProperty)
                .map(modelProperty -> new OrderBy(modelProperty,
                        QuerySpecUtil.crnkDirectionToOrderByDirection(sortSpec.getDirection()))
                );
    }

    @JsonFormat(shape = Shape.ARRAY)
    public static class SortNode {
         private String field;
         private Integer direction;

        @JsonCreator
        public SortNode(@JsonProperty(value = "field", required = true) String field,
                        @JsonProperty("direction") Integer direction) {
            this.field = field;
            this.direction = direction;
        }

        public String getField() {
            return field;
        }

        public void setField(String field) {
            this.field = field;
        }

        public Integer getDirection() {
            return direction;
        }

        public void setDirection(Integer direction) {
            this.direction = direction;
        }
    }

    public Function<String, List<OrderBy>> customSortToOrderByConverter() {
        return this::customSortToOrderBy;
    }

    private List<OrderBy> customSortToOrderBy(String queryParam) {
        JsonNode outerNode;
        try {
            outerNode = objectMapper.readTree(queryParam);
        } catch (JsonProcessingException e) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS);
        }

        List<SortNode> sortNodes = new ArrayList<>();
        if (outerNode.isArray()) {
            for (var jsonNode: outerNode) {
                try {
                    SortNode sortNode;
                    if (jsonNode.isArray()) { // [["field1", 1], ["field2", 0]]
                        sortNode = objectMapper.convertValue(jsonNode, SortNode.class);
                        if (!(sortNode.direction.equals(1) || sortNode.direction.equals(0))) {
                            throw new IllegalArgumentException("Direction must be 1 or 0");
                        }
                    } else if (jsonNode.isTextual()) { // ["field1", "field2"]
                        String fieldName = objectMapper.convertValue(jsonNode, String.class);
                        sortNode = new SortNode(fieldName, 0); // default direction ASC
                    } else {
                        throw new IllegalArgumentException("Json node type " +
                                jsonNode.getNodeType() + " not supported for conversion to PageSortNode");
                    }
                    sortNodes.add(sortNode);
                } catch (IllegalArgumentException e) {
                    throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS);
                }
            }
        }

        return sortNodes.stream()
                .map(this::sortNodeToOrderBy)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    private Optional<OrderBy> sortNodeToOrderBy(SortNode sortNode) {
        Direction dir = sortNode.direction == 0 ? ASC : DESC;
        return Optional.ofNullable(sortNode.field)
                .map(apiFieldMap::get)
                .map(ApiField::getModelPath)
                .map(ModelPathForApi::rootProperty)
                .map(prop -> new OrderBy(prop, dir));
    }

    @Override
    public MetaInformation getMetaInformation(Collection<Resource> resources, QuerySpec querySpec,
                                              MetaInformation current) {
        if (current != null || HTTP_POST.equals(httpRequestContextProvider.getRequestContext().getMethod())) {
            return current;
        }
        if (httpRequestContextProvider.getRequestContext().getMethod().equals(HttpMethod.GET.name())) {
            return crnkAdditionalDataService.fillBodyMetaInformation(new BaseCrnkMetaInformation(), querySpec,
                    apiService);
        } else {
            return crnkAdditionalDataService.fillBodyMetaInformation(new BaseCrnkMetaInformation(), resources);
        }
    }

    /**
     * Заполняет links в body
     * findOne и save умеет заполнять links только в body.data
     */
    @Override
    public LinksInformation getLinksInformation(Collection<Resource> resources, QuerySpec querySpec,
                                                LinksInformation current) {

        if (current != null) {
            return current;
        } else if (HTTP_POST.equals(httpRequestContextProvider.getRequestContext().getMethod())) {
            return new EmptyLinksInformation();
        }

        // TODO remove and regenerate tests
        boolean allowAddFields = !HTTP_PATCH.equals(httpRequestContextProvider.getRequestContext().getMethod());

        Result<HttpRequestContext> result = httpRequestContextProvider.getRequestContextResult();
        HttpRequestContext httpRequestContext = result.get();

        BaseCrnkLinksInformation linksInformation = crnkAdditionalDataService.getBodyLinksInformation(querySpec,
                httpRequestContext, apiService, allowAddFields);

        if (SELF_LINKING_METHODS.contains(HttpMethod.resolve(httpRequestContext.getMethod()))) {
            resources.stream()
                    .filter(Objects::nonNull)
                    .map(resource ->
                            crnkAdditionalDataService.selfLink(httpRequestContext, resource.getId())
                                    // TODO it's uncommon for post/patch to have query params
                                    + crnkAdditionalDataService.queryString(querySpec)
                    )
                    .findFirst()
                    .ifPresent(linksInformation::setSelf);
        }

        return linksInformation;
    }

    private void setSelfLink(Resource resource) {
        if (apiService.needToSetSelfLink()) {
            HttpRequestContext httpRequestContext = httpRequestContextProvider.getRequestContext();
            LinksInformation baseCrnkLinksInformation =
                    crnkAdditionalDataService.selfLinksInformation(resource, httpRequestContext);
            resource.setLinks(objectMapper.valueToTree(baseCrnkLinksInformation));
        }
    }

    private FilterNode getFilterNode(QuerySpec querySpec) {
        // Пытаемся достать CrnkFilterAdapter, если не получается пытаемся достать FilterSpec
        FilterNode filterNode = QuerySpecUtil.getCrnkFilterAdapter(querySpec)
                .map(CrnkFilterAdapter::getFilterNode)
                .orElse(null);

        if (filterNode != null) {
            return filterNode;
        } else {
            for (FilterSpec filter : querySpec.getFilters()) {
                if (filter.getValue() != null) {
                    return FilterNode.fromFilterSpec(filter);
                }
            }
        }

        return null;
    }

    public ApiService<M> getApiService() {
        return apiService;
    }

    @FunctionalInterface
    private interface Persister<M extends ModelWithId> {
        M persist(M parsedModel, IncomingApiFields<M> incomingApiFields, List<DefectInfo<Defect>> parseDefectInfos);
    }
}
