package ru.yandex.partner.jsonapi.crnk;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

import io.crnk.core.queryspec.Direction;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.queryspec.pagingspec.NumberSizePagingBehavior;
import io.crnk.core.queryspec.pagingspec.NumberSizePagingSpec;
import io.crnk.core.queryspec.pagingspec.OffsetLimitPagingSpec;
import io.crnk.core.queryspec.pagingspec.PagingBehavior;

import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilterAdapter;

public class QuerySpecUtil {

    private QuerySpecUtil() {
    }

    public static Optional<LimitOffset> getLimitOffset(QuerySpec querySpec) {
        Optional<OffsetLimitPagingSpec> paging = Optional.ofNullable(querySpec.getPaging(OffsetLimitPagingSpec.class));
        Optional<Integer> limitOpt = paging.map(OffsetLimitPagingSpec::getLimit).map(Long::intValue);
        int offset = paging.map(OffsetLimitPagingSpec::getOffset).map(Long::intValue).orElse(0);
        return limitOpt.map(limit -> LimitOffset.limited(limit, offset));
    }

    public static Optional<CrnkFilterAdapter> getCrnkFilterAdapter(QuerySpec querySpec) {
        return QuerySpecUtil.getFilterSpecByClass(querySpec, CrnkFilterAdapter.class);
    }

    public static Optional<CustomParametersHolder> getCustomParametersHolder(QuerySpec querySpec) {
        return QuerySpecUtil.getFilterSpecByClass(querySpec, CustomParametersHolder.class);
    }

    public static CustomParametersHolder computeCustomParametersHolder(QuerySpec querySpec) {
        return getCustomParametersHolder(querySpec)
                .orElseGet(() -> {
                    CustomParametersHolder parametersHolder = new CustomParametersHolder();
                    querySpec.addFilter(parametersHolder);
                    return parametersHolder;
                });
    }

    public static void setTotalRequested(QuerySpec querySpec) {
        computeCustomParametersHolder(querySpec).setTotalRequested();
    }

    public static boolean isTotalRequested(QuerySpec querySpec) {
        return getCustomParametersHolder(querySpec)
                .map(CustomParametersHolder::isTotalRequested)
                .orElse(false);
    }

    public static void setOriginalParameters(QuerySpec querySpec, Map<String, Set<String>> parameters) {
        computeCustomParametersHolder(querySpec).setOriginalParameters(parameters);
    }

    public static Map<String, Set<String>> getOriginalParameters(QuerySpec querySpec) {
        return getCustomParametersHolder(querySpec)
                .map(CustomParametersHolder::getOriginalParameters)
                .orElse(Map.of());
    }

    protected static <T extends FilterSpec> Optional<T> getFilterSpecByClass(QuerySpec querySpec, Class<T> clazz) {
        return querySpec.getFilters().stream().filter(clazz::isInstance).findFirst().map(clazz::cast);
    }

    public static OrderBy.Direction crnkDirectionToOrderByDirection(Direction direction) {
        return Direction.ASC.equals(direction) ? OrderBy.Direction.ASC : OrderBy.Direction.DESC;
    }

    /**
     * Нужно чтобы заработало корректное pagingBehavior
     * {@link io.crnk.core.queryspec.QuerySpec#QuerySpec(Class, String)} - if-чик на Resource.class
     * {@link io.crnk.core.queryspec.mapper.DefaultQuerySpecUrlMapper#serialize(QuerySpec, Map, QuerySpec)}
     * {@code RegistryEntry rootEntry = resourceRegistry.getEntry(rootQuerySpec.getResourceClass());
     *        if (rootEntry != null && rootEntry.getResourceInformation() != null
     *             && rootEntry.getResourceInformation().getPagingSpecType() != null) }
     */
    public static Map<String, Set<String>> getPageBehaviorParameters(QuerySpec querySpec) {
        PagingBehavior<NumberSizePagingSpec> pagingBehavior = new NumberSizePagingBehavior();
        NumberSizePagingSpec pagingSpec = querySpec.getPaging(NumberSizePagingSpec.class);
        return pagingBehavior.serialize(pagingSpec, querySpec.getResourceType());
    }
}
