package ru.yandex.partner.jsonapi.crnk.authorization.actions;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.action.Action;
import ru.yandex.partner.jsonapi.crnk.action.check.ActionChecker;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.libs.auth.facade.UserAuthenticationProvider;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;
import ru.yandex.partner.libs.utils.StreamUtils;

@ParametersAreNonnullByDefault
public abstract class AbstractActionsAuthorizationService<M> implements ActionsAuthorizationService<M> {

    private static final String ACTION_RIGHT_PREFIX = "do_";

    private final ApiModelMetaData<M> apiModelMetaData;
    private final MultistateGraph<? super M, ?> graph;
    private final UserAuthenticationProvider authenticationFacade;
    private final List<ModelProperty<? super M, ?>> requiredProperties;

    protected AbstractActionsAuthorizationService(@Nullable MultistateGraph<? super M, ?> graph,
                                                  ApiModelMetaData<M> apiModelMetaData,
                                                  UserAuthenticationProvider authenticationFacade,
                                                  List<ModelProperty<? super M, ?>> requiredProperties) {
        this.graph = graph;
        this.apiModelMetaData = apiModelMetaData;
        this.authenticationFacade = authenticationFacade;
        this.requiredProperties =
                Stream.concat(
                        requiredProperties.stream(),
                        Optional.ofNullable(graph)
                                .map(MultistateGraph::getAllRequiredProperties)
                                .map(Set::stream)
                                .orElse(Stream.of())
                ).distinct().toList();
    }

    protected abstract Map<String, ActionAuthEntry<M>> getAuthEntryMap();

    @Override
    public List<ModelProperty<? super M, ?>> getAllRequiredProperties() {
        return requiredProperties;
    }

    @Override
    public List<Boolean> checkActionAuthorized(String actionName, List<M> coreModels) {
        UserAuthentication ua = authenticationFacade.getUserAuthentication();
        ActionAuthEntry<M> authEntry = getAuthEntryMap().get(actionName);
        return authEntry == null ? Collections.nCopies(coreModels.size(), false)
                : authorizeWithEntry(ua, authEntry, coreModels);
    }

    @Override
    public boolean checkActionAuthorized(String actionName, M coreModel) {
        return checkActionAuthorized(actionName, List.of(coreModel)).get(0);
    }

    @Override
    public boolean checkActionAllowedAndAuthorized(String actionName, M coreModel) {
        if (!graph.checkActionAllowed(actionName, coreModel)) {
            return false;
        }

        return checkActionAuthorized(actionName, coreModel);
    }

    @Override
    public Set<Action> filterApiActions(M model, Set<Action> allowedActions) {
        UserAuthentication ua = authenticationFacade.getUserAuthentication();

        return allowedActions.stream()
                .filter(action -> {
                    ActionAuthEntry<M> authEntry = getAuthEntryMap().get(action.getName());
                    if (authEntry == null || !authEntry.isExposed()) {
                        return false;
                    } else {
                        return authorizeWithEntry(ua, authEntry, List.of(model)).get(0);
                    }
                })
                .collect(Collectors.toSet());
    }

    @Override
    public Map<String, String> getRightNamesForActions() {
        return getAuthEntryMap().entrySet().stream()
                .filter(e -> e.getValue().isRightRequired())
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> getRightName(e.getValue())
                ));
    }

    protected ApiModelMetaData<M> getApiModelMetaData() {
        return apiModelMetaData;
    }

    /**
     * Проверяет, может ли пользователь из UserAuthentication выполнять действие authEntry над моделями models\
     * <p>
     * Этот метод можно переопределять в потомках для добавления кастомных проверок, общих для всех моделей
     */
    protected List<Boolean> authorizeWithEntry(
            UserAuthentication ua, ActionAuthEntry<M> authEntry, List<M> coreModels) {

        if (authEntry.isRightRequired() && !ua.userHasRight(getRightName(authEntry))) {
            return Collections.nCopies(coreModels.size(), false);
        }

        if (authEntry.getChecks() != null) {
            List<List<Boolean>> checkResults = authEntry.getChecks().stream()
                    .map(check -> check.check(ua, apiModelMetaData.getResourceType(), coreModels))
                    .collect(Collectors.toList());

            return StreamUtils.reduceLists(
                    Collections.nCopies(coreModels.size(), true),
                    checkResults,
                    Boolean::logicalAnd
            );
        }

        return Collections.nCopies(coreModels.size(), true);
    }

    @Nonnull
    private String getRightName(ActionAuthEntry<M> authEntry) {
        return ACTION_RIGHT_PREFIX + authEntry.getRightPrefix() + "_" + authEntry.getActionName();
    }

    @Override
    public boolean isActionExposed(String actionName) {
        var action = getAuthEntryMap().get(actionName);
        return action != null && action.isExposed();
    }

    @Override
    public Set<ModelProperty<? extends Model, ?>> getRequiredPropertiesByAction(ActionNameHolder actionNameHolder) {
        var actionName = actionNameHolder.getActionName();

        var actionAuthEntry = getAuthEntryMap().get(actionName);
        if (actionAuthEntry == null) {
            throw new IllegalArgumentException("Unknown action: " + actionName);
        }

        return actionAuthEntry.getChecks().stream()
                .map(ActionChecker::getRequiredProperties)
                .flatMap(Collection::stream).map(property -> (ModelProperty<? extends Model, ?>) property)
                .collect(Collectors.toSet());
    }
}
