package ru.yandex.partner.jsonapi.crnk.block.authorization.actions;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.AbstractActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionAuthEntry;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

@ParametersAreNonnullByDefault
public abstract class BlockActionsAuthorizationService<B extends BaseBlock>
        extends AbstractActionsAuthorizationService<B> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BlockActionsAuthorizationService.class);

    public BlockActionsAuthorizationService(MultistateGraph<B, BlockStateFlag> graph,
                                            ApiModelMetaData<B> apiModelMetaData,
                                            AuthenticationFacade authenticationFacade,
                                            BlockActionsCheckers<B> blockActionsCheckers) {
        super(graph, apiModelMetaData, authenticationFacade, blockActionsCheckers.getAllRequiredFields());
    }

    @Override
    protected List<Boolean> authorizeWithEntry(UserAuthentication ua,
                                               ActionAuthEntry<B> authEntry,
                                               List<B> coreModels) {
        Optional<Long> protectedPageId = coreModels.stream()
                .filter(this::isPageProtected)
                .map(this::extractPageId)
                .findAny();

        if (protectedPageId.isPresent()) {
            LOGGER.info(
                    "User [ %s ] has tried to do action [ %s ] on PROTECTED page [ %s ]".formatted(
                            ua.getLogin(),
                            authEntry.getActionName(),
                            protectedPageId.get()
                    )
            );
            if (!ua.userHasRight(CoreConstants.EDIT_PROTECTED_PAGES_RIGHT_NAME)) {
                LOGGER.info(
                        "User [ %s ] has no rights to do action [ %s ] on PROTECTED page [ %s ]".formatted(
                                ua.getLogin(),
                                authEntry.getActionName(),
                                protectedPageId.get()
                        )
                );
                return Collections.nCopies(coreModels.size(), false);
            }
            // is_my should be checked on Policy level
        } else if (!ua.userHasRight(BlockRightNames.EDIT.getFullRightName(getApiModelMetaData().getResourceType()))) {
            return Collections.nCopies(
                    coreModels.size(),
                    canEditAll(ua)
            );
        }
        return super.authorizeWithEntry(ua, authEntry, coreModels);
    }

    /**
     * Используется только на внутренних типах блоков
     */
    protected boolean canEditAll(UserAuthentication ua) {
        return false;
    }

    protected abstract boolean isPageProtected(B block) throws RuntimeException;

    protected abstract Long extractPageId(B block);
}
