package ru.yandex.partner.jsonapi.crnk.block.rtb.authorization;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.page.model.PageWithMultistate;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.get.ActiveBlockRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.get.BlockByPageRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.get.BlockViewRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.get.TechnicalBlockRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.get.ZeroBlockRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.post.EditProtectedPageRule;
import ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.post.HasAddRule;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.authorization.actioncontext.CreateRequest;
import ru.yandex.partner.libs.authorization.actioncontext.CreateResult;
import ru.yandex.partner.libs.authorization.actioncontext.CustomRequestContext;
import ru.yandex.partner.libs.authorization.actioncontext.CustomResult;
import ru.yandex.partner.libs.authorization.actioncontext.GetAllResult;
import ru.yandex.partner.libs.authorization.actioncontext.GetResult;
import ru.yandex.partner.libs.authorization.actioncontext.RequestContext;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.authorization.policy.base.Policy;

public abstract class BlockAuthorizationPolicy<B extends BaseBlock> implements Policy<B> {

    private final ApiModelMetaData<B> apiModelMetaData;
    private final BlockViewRule blockViewRule;
    private final ActiveBlockRule activeBlockRule;
    private final BlockByPageRule<B> blockByPageRule;
    private final TechnicalBlockRule technicalBlockRule;
    private final ZeroBlockRule zeroBlockRule;
    private final HasAddRule hasAddRule;
    private final EditProtectedPageRule editProtectedPageRule;

    @SuppressWarnings("checkstyle:parameternumber")
    public BlockAuthorizationPolicy(
            ApiModelMetaData<B> apiModelMetaData,
            BlockViewRule blockViewRule,
            ActiveBlockRule activeBlockRule,
            BlockByPageRule<B> blockByPageRule,
            TechnicalBlockRule technicalBlockRule,
            ZeroBlockRule zeroBlockRule,
            HasAddRule hasAddRule,
            EditProtectedPageRule editProtectedPageRule
    ) {

        this.apiModelMetaData = apiModelMetaData;
        this.blockViewRule = blockViewRule;
        this.activeBlockRule = activeBlockRule;
        this.blockByPageRule = blockByPageRule;
        this.technicalBlockRule = technicalBlockRule;
        this.zeroBlockRule = zeroBlockRule;
        this.hasAddRule = hasAddRule;
        this.editProtectedPageRule = editProtectedPageRule;
    }

    @Override
    public AuthorizationDecision<B> authorizeCreateRequest(CreateRequest<B> createRequest) {
        String modelName = apiModelMetaData.getResourceType();
        return AuthorizationDecision.denyOverrides(
                hasAddRule.authorize(
                        modelName,
                        new RequestContext(createRequest.getUserAuthentication())
                ),
                editProtectedPageRule.authorize(createRequest, getPageClass())
        );
    }

    @Override
    public AuthorizationDecision<B> authorizeCreateResult(CreateResult<B> createResult) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<B> authorizeGetAllResult(GetAllResult<B> getAllResult) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<B> authorizeGetRequest(RequestContext getRequest) {
        UserAuthentication userAuthentication = getRequest.getUserAuthentication();
        return AuthorizationDecision.firstApplicable(
                blockViewRule.authorize(apiModelMetaData.getResourceType(), userAuthentication),
                AuthorizationDecision.denyOverrides(
                        activeBlockRule.authorize(),
                        blockByPageRule.authorize(getRequest),
                        zeroBlockRule.authorize(apiModelMetaData.getResourceType(), userAuthentication),
                        technicalBlockRule.authorize(apiModelMetaData.getResourceType(), userAuthentication)
                )
        );
    }

    @Override
    public AuthorizationDecision<B> authorizeGetResult(GetResult<B> getResult) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<B> authorizeCustomRequest(CustomRequestContext customRequest) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<B> authorizeCustomResult(CustomResult<B> customResult) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<B> authorizeUpdateRequest(RequestContext updateRequest) {
        return AuthorizationDecision.permit();
    }

    public abstract Class<? extends PageWithMultistate> getPageClass();
}
