package ru.yandex.partner.jsonapi.crnk.block.rtb.authorization.post;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BlockValidationMsg;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.PageWithMultistate;
import ru.yandex.partner.core.entity.page.service.PageService;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.authorization.actioncontext.CreateRequest;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;

@Component
public class EditProtectedPageRule {

    private static final Logger LOGGER = LoggerFactory.getLogger(EditProtectedPageRule.class);
    private final PageService pageService;

    public EditProtectedPageRule(PageService pageService) {
        this.pageService = pageService;
    }

    public <T extends BaseBlock> AuthorizationDecision<T> authorize(
            CreateRequest<T> createRequest,
            Class<? extends PageWithMultistate> pageClass
    ) {
        UserAuthentication ua = createRequest.getUserAuthentication();

        if (!ua.userHasRight(CoreConstants.EDIT_PROTECTED_PAGES_RIGHT_NAME)) {
            T block = createRequest.getCreatePayload();
            if (block.getPageId() == null) {
                throw new CrnkResponseStatusException(
                        HttpErrorStatusEnum.ERROR__PARAMS,
                        BlockValidationMsg.PAGE_ID_EXPECTED
                );
            }

            var isProtected = pageService.findAll(QueryOpts.forClass(pageClass)
                            .withProps(PageWithMultistate.MULTISTATE)
                            .withFilter(PageFilters.PAGE_ID.eq(block.getPageId()))
                    ).stream()
                    .findFirst()
                    .map(PageWithMultistate::getMultistate)
                    .map(m -> m.hasFlag(PageStateFlag.PROTECTED))
                    .orElseThrow(() -> new CrnkResponseStatusException(
                            HttpErrorStatusEnum.ERROR__NOT_FOUND,
                            BlockValidationMsg.PAGE_ID_UNREACHABLE.format(block.getPageId()))
                    );

            if (isProtected) {
                return AuthorizationDecision.deny(
                        "You don't have rights to add block on protected page %s".formatted(block.getPageId())
                );
            }
            return AuthorizationDecision.permit();
        }
        LOGGER.info("User {} with uid {} has passed EditProtectedRule", ua.getLogin(), ua.getUid());
        return AuthorizationDecision.permit();
    }
}
