package ru.yandex.partner.jsonapi.crnk.block.rtb.external.authorization.actions;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionAuthEntry;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.BlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.BlockActionsCheckers;
import ru.yandex.partner.jsonapi.models.block.rtb.external.ApiRtbMetaData;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

@Service
public class RtbBlockActionsAuthorizationService extends BlockActionsAuthorizationService<RtbBlock> {

    private final BlockActionsCheckers<RtbBlock> blockActionsCheckers;

    @Autowired
    public RtbBlockActionsAuthorizationService(ApiRtbMetaData apiModelMetaData,
                                               MultistateGraph<RtbBlock, BlockStateFlag> graph,
                                               AuthenticationFacade authenticationFacade,
                                               BlockActionsCheckers<RtbBlock> blockActionsCheckers) {
        super(graph, apiModelMetaData, authenticationFacade, blockActionsCheckers);
        this.blockActionsCheckers = blockActionsCheckers;
    }

    @Override
    protected Map<String, ActionAuthEntry<RtbBlock>> getAuthEntryMap() {
        return ActionAuthEntry.buildMap(getApiModelMetaData().getResourceType(), List.of(
                ActionAuthEntry.<RtbBlock>builder()
                        .setActionName(BlockActionsEnum.DELETE)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.delete()),
                ActionAuthEntry.<RtbBlock>builder()
                        .setActionName(BlockActionsEnum.RESTORE)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.restore()),
                ActionAuthEntry.<RtbBlock>builder()
                        .setActionName(BlockActionsEnum.EDIT)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.edit()),
                ActionAuthEntry.<RtbBlock>builder()
                        .setActionName(BlockActionsEnum.DUPLICATE)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.duplicate())
        ));
    }

    @Override
    protected boolean isPageProtected(RtbBlock block) {
        return Optional.of(block)
                .map(RtbBlock::getCampaign)
                .map(ContextPage::getMultistate)
                .map(m -> m.hasFlag(PageStateFlag.PROTECTED))
                .orElseThrow(() -> new RuntimeException(
                        ("Can't check is page protected on block: " +
                                "page or page's multistate is not loaded. Block id [ %s ]")
                                .formatted(block.getId())
                ));
    }

    @Override
    protected Long extractPageId(RtbBlock block) {
        return block.getCampaign().getId();
    }
}
