package ru.yandex.partner.jsonapi.crnk.block.rtb.internal.authorization.actions;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionAuthEntry;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.BlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.BlockActionsCheckers;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.rtb.internal.ApiInternalRtbMetaData;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

@Service
public class InternalRtbBlockActionsAuthorizationService extends BlockActionsAuthorizationService<InternalRtbBlock> {

    private final BlockActionsCheckers<InternalRtbBlock> blockActionsCheckers;

    @Autowired
    public InternalRtbBlockActionsAuthorizationService(ApiInternalRtbMetaData apiModelMetaData,
                                                       MultistateGraph<InternalRtbBlock, BlockStateFlag> graph,
                                                       AuthenticationFacade authenticationFacade,
                                                       BlockActionsCheckers<InternalRtbBlock> blockActionsCheckers) {
        super(graph, apiModelMetaData, authenticationFacade, blockActionsCheckers);
        this.blockActionsCheckers = blockActionsCheckers;
    }

    @Override
    protected Map<String, ActionAuthEntry<InternalRtbBlock>> getAuthEntryMap() {
        return ActionAuthEntry.buildMap(getApiModelMetaData().getResourceType(), List.of(
                ActionAuthEntry.<InternalRtbBlock>builder()
                        .setActionName(BlockActionsEnum.DELETE)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.delete()),
                ActionAuthEntry.<InternalRtbBlock>builder()
                        .setActionName(BlockActionsEnum.RESTORE)
                        .setExposed(true)
                        .setRightRequired(true) // !
                        .setChecks(blockActionsCheckers.restore()),
                ActionAuthEntry.<InternalRtbBlock>builder()
                        .setActionName(BlockActionsEnum.EDIT)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.edit()),
                ActionAuthEntry.<InternalRtbBlock>builder()
                        .setActionName(BlockActionsEnum.DUPLICATE)
                        .setExposed(true)
                        .setRightRequired(false)
                        .setChecks(blockActionsCheckers.duplicate())
        ));
    }

    @Override
    protected boolean canEditAll(UserAuthentication ua) {
        return ua.userHasRight(BlockRightNames.EDIT_ALL.getFullRightName(getApiModelMetaData().getResourceType()));
    }

    @Override
    protected boolean isPageProtected(InternalRtbBlock block) {
        return Optional.of(block)
                .map(InternalRtbBlock::getCampaign)
                .map(InternalContextPage::getMultistate)
                .map(m -> m.hasFlag(PageStateFlag.PROTECTED))
                .orElseThrow(() -> new RuntimeException(
                        "Can't check is page protected on block: multistate is not loaded. Block id %s"
                                .formatted(block.getId())
                ));
    }

    @Override
    protected Long extractPageId(InternalRtbBlock block) {
        return block.getCampaign().getId();
    }
}
