package ru.yandex.partner.jsonapi.crnk.exceptions.mapper;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nonnull;

import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.engine.error.ExceptionMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.jsonapi.response.ExceptionToErrorIdMapper;
import ru.yandex.partner.libs.i18n.GettextMsg;

@Component
public class CrnkExceptionMapper implements ExceptionMapper<Exception>, MessageSourceAware {
    public static final Logger LOGGER = LoggerFactory.getLogger(CrnkExceptionMapper.class);
    protected MessageSourceAccessor messages = new MessageSourceAccessor(new DelegatingMessageSource());

    private final HttpStatus httpStatus;
    private final String errorId;
    private final String code;
    private final GettextMsg titleMsg;
    private final ErrorDataWithProfile errorDataWithProfile;

    @Autowired
    public CrnkExceptionMapper(ExceptionToErrorIdMapper exceptionToErrorIdMapper,
                               ErrorDataWithProfile errorDataWithProfile) {
        httpStatus = HttpStatus.INTERNAL_SERVER_ERROR;

        final int errorId0 = exceptionToErrorIdMapper.mapHttpStatusToErrorId(httpStatus.value());
        errorId = String.valueOf(errorId0);
        code = String.valueOf(httpStatus.value());

        titleMsg = Optional.ofNullable(JsonapiErrorMsg.forErrorId(errorId0))
                .orElseGet(() -> JsonapiErrorMsg.forStatus(httpStatus));

        this.errorDataWithProfile = errorDataWithProfile;
    }

    @Override
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        this.messages = new MessageSourceAccessor(messageSource);
    }

    @Override
    public ErrorResponse toErrorResponse(Exception exception) {
        LOGGER.error(exception.getMessage(), exception);

        String title = titleMsg == null
                ? httpStatus.getReasonPhrase()
                : messages.getMessage(titleMsg);

        ErrorData errorData = errorDataWithProfile.getErrorData(exception, errorId, code, title);

        return ErrorResponse.builder()
                .setStatus(httpStatus.value())
                .setErrorData(List.of(errorData))
                .build();
    }

    @Override
    public Exception fromErrorResponse(ErrorResponse errorResponse) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean accepts(ErrorResponse errorResponse) {
        return false;
    }
}
