package ru.yandex.partner.jsonapi.crnk.fields;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import io.crnk.core.engine.information.resource.ResourceFieldType;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.partner.jsonapi.jackson.CollectingErrorsParser;
import ru.yandex.partner.jsonapi.models.ParametrizedJavaTypeHolder;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.utils.StreamUtils;

/**
 * Описания полей можно посмотреть тут {@link ApiFieldBuilder}
 *
 * @param <M> Модель из core слоя
 */
public class ApiField<M extends Model> {
    private final String jsonName;
    private final ModelPathForApi<? super M, ?> modelPath;
    private final BatchFunction<M, Object> getterApiValue;
    private final BatchBiFunction<M, Object> setterCoreValue;
    private final Consumer<ApiFieldsAccessRules.Builder<M>> availableFunction;
    private final Consumer<ApiFieldsAccessRules.Builder<EditableData<M>>> addFieldsFunction;
    private final Consumer<ApiFieldsAccessRules.Builder<EditableData<M>>> editableFunction;
    private final Consumer<QueryParamsContext<M>> queryParamsContextProcessor;
    private final Function<QueryParamsContext<M>, Map<String, Object>> defaultsFunction;
    private final List<ModelProperty<? super M, ?>> requiredProperties;
    private final List<CheckInnerField> checkInnerFields;
    private final ResourceFieldType resourceFieldType;
    private final JavaType javaType;
    private final String deprecated;

    <AV, CV> ApiField(ApiFieldBuilder<M, AV, CV> apiFieldBuilder) {
        this.jsonName = apiFieldBuilder.getJsonName();
        this.modelPath = apiFieldBuilder.getModelPath();
        this.availableFunction = apiFieldBuilder.getAvailableBuilderBiConsumer() != null
                ? builder -> apiFieldBuilder.getAvailableBuilderBiConsumer().accept(jsonName, builder)
                : builder -> builder.always(jsonName);
        this.addFieldsFunction = apiFieldBuilder.getAddFieldsBuilderBiConsumer() != null
                ? builder -> apiFieldBuilder.getAddFieldsBuilderBiConsumer().accept(jsonName, builder)
                : null;
        this.editableFunction =
                apiFieldBuilder.getEditableBuilderBiConsumer() != null
                        ? builder -> apiFieldBuilder.getEditableBuilderBiConsumer().accept(jsonName, builder)
                        : null;
        this.requiredProperties = Objects.requireNonNullElse(apiFieldBuilder.getRequiredProperties(), List.of());
        this.checkInnerFields = StreamUtils.mergedStream(apiFieldBuilder.getInnerFieldPaths(),
                apiFieldBuilder.getInnerFieldRights(),
                (jsonPath, crnkModelRight) -> new CheckInnerField(jsonName, jsonPath, crnkModelRight))
                .collect(Collectors.toList());
        this.resourceFieldType = apiFieldBuilder.getResourceFieldType();
        this.javaType = apiFieldBuilder.getJavaType();
        //noinspection unchecked
        this.getterApiValue = (BatchFunction<M, Object>) apiFieldBuilder.getGetterApiValue();
        //noinspection unchecked
        this.setterCoreValue = (BatchBiFunction<M, Object>) apiFieldBuilder.getSetterCoreValue();
        this.defaultsFunction = apiFieldBuilder.getDefaultsFunction();
        this.queryParamsContextProcessor = apiFieldBuilder.getExtraQueryParamsConsumer();
        this.deprecated = apiFieldBuilder.getDeprecated();
    }

    public String getJsonName() {
        return jsonName;
    }

    public ModelPathForApi<? super M, ?> getModelPath() {
        return modelPath;
    }

    public Consumer<ApiFieldsAccessRules.Builder<M>> getAvailableFunction() {
        return availableFunction;
    }

    @Nullable
    public Consumer<ApiFieldsAccessRules.Builder<EditableData<M>>> getAddFieldsFunction() {
        return addFieldsFunction;
    }

    @Nullable
    public Consumer<ApiFieldsAccessRules.Builder<EditableData<M>>> getEditableFunction() {
        return editableFunction;
    }

    public List<ModelProperty<? super M, ?>> getRequiredProperties() {
        return requiredProperties;
    }

    public List<CheckInnerField> getCheckInnerFields() {
        return checkInnerFields;
    }

    public ResourceFieldType getResourceFieldType() {
        return resourceFieldType;
    }

    public JavaType getValueClass() {
        return javaType;
    }

    public Object toApiValue(M model) {
        return getterApiValue.applyOne(model);
    }

    public Object toApiValues(List<M> models) {
        return getterApiValue.applyList(models);
    }

    public boolean hasCoreValue() {
        return setterCoreValue != null;
    }

    public Map<String, Object> getDefaults(QueryParamsContext<M> queryParamsContext) {
        if (defaultsFunction == null) {
            return Collections.emptyMap();
        }
        return defaultsFunction.apply(queryParamsContext);
    }

    public void consumeExtraQueryParams(QueryParamsContext<M> queryParamsContext) {
        if (queryParamsContextProcessor != null && queryParamsContext.getRawParams() != null) {
            queryParamsContextProcessor.accept(queryParamsContext);
        }
    }

    public List<DefectInfo<Defect>> tryFillCoreValue(M model, JsonNode jsonNode, CollectingErrorsParser parser)
            throws IOException {
        Object value = parser.parse(jsonNode, javaType);

        List<DefectInfo<Defect>> defects = parser.getAndClearCollectedDefects(this.getJsonName());
        if (!defects.isEmpty()) {
            return defects;
        }

        try {
            var defectInfos = setterCoreValue.acceptOne(model, value);
            if (defectInfos != null && !defectInfos.isEmpty()) {
                PathConverter prependJsonName = PathConverter.prepend(PathHelper.pathFromStrings(jsonName));

                return defectInfos.stream()
                        .map(info -> (DefectInfo<?>) info)
                        .map(defectInfo -> (DefectInfo<Defect>) defectInfo.convertPath(prependJsonName))
                        .collect(Collectors.toList());
            } else {
                return null;
            }
        } catch (Exception e) {
            // здесь можно вернуть дефолтный дефект и вернуть 400
            // но наверное лучше выкинуть исключение чтобы такие места быстрее отлавливать ???
            throw new IllegalStateException("Something went wrong during the conversion. " +
                    "ApiField.jsonName = " + jsonName, e);
        }
    }

    @Nullable
    public String getDeprecated() {
        return deprecated;
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое есть только в api
     */
    public static <T extends Model, V> ApiFieldBuilder<T, V, Void> apiField(Class<V> avClass,
                                                                            BatchFunction<T, V> getter) {
        return new ApiFieldBuilder<>(avClass, (ModelPathForApi<? super T, Void>) null, getter, null);
    }

    public static <T extends Model, V> ApiFieldBuilder<T, V, Void> apiField(ParametrizedJavaTypeHolder<V> avClass,
                                                                            BatchFunction<T, V> getter) {
        return new ApiFieldBuilder<>(avClass, (ModelPathForApi<? super T, Void>) null, getter, null);
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения
     */
    public static <T extends Model, V> ApiFieldBuilder<T, V, V> readableApiField(
            Class<V> avClass,
            ModelProperty<? super T, V> modelProperty) {
        return new ApiFieldBuilder<>(avClass,
                modelProperty,
                BatchFunction.one(modelProperty::get),
                null);
    }

    public static <T extends Model, V> ApiFieldBuilder<T, V, V> readableApiField(
            ParametrizedJavaTypeHolder<V> avClass,
            ModelProperty<? super T, V> modelProperty) {
        return new ApiFieldBuilder<>(avClass,
                modelProperty,
                BatchFunction.one(modelProperty::get),
                null);
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения
     * с кастомным проебразованием
     */
    public static <T extends Model, AV, CV> ApiFieldBuilder<T, AV, CV> readableApiField(
            Class<AV> avClass,
            ModelProperty<? super T, CV> modelProperty,
            BatchFunction<T, AV> getterApiValue) {
        return new ApiFieldBuilder<>(avClass,
                modelProperty,
                getterApiValue,
                null);
    }

    public static <T extends Model, AV, CV> ApiFieldBuilder<T, AV, CV> readableApiField(
            ParametrizedJavaTypeHolder<AV> avClass,
            ModelProperty<? super T, CV> modelProperty,
            BatchFunction<T, AV> getterApiValue) {
        return new ApiFieldBuilder<>(avClass,
                modelProperty,
                getterApiValue,
                null);
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения и записи
     */
    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            Class<V> avClass,
            ModelProperty<? super T, V> modelProperty) {

        return new ApiFieldBuilder<>(avClass,
                ModelPath.of(modelProperty),
                BatchFunction.one(modelProperty::get),
                BatchBiFunction.byModelProperty(modelProperty));
    }

    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            ParametrizedJavaTypeHolder<V> avClass,
            ModelProperty<? super T, V> modelProperty) {

        return new ApiFieldBuilder<>(avClass,
                ModelPath.of(modelProperty),
                BatchFunction.one(modelProperty::get),
                BatchBiFunction.byModelProperty(modelProperty));
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения и записи
     * для вложенных объектов
     */
    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            Class<V> avClass,
            ModelPathForApi<? super T, V> modelPath) {
        return new ApiFieldBuilder<>(
                avClass, modelPath,
                BatchFunction.one(modelPath::get),
                BatchBiFunction.one((model, value) -> {
                    modelPath.set(model, value);
                    return null;
                })
        );
    }

    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            ParametrizedJavaTypeHolder<V> avClass,
            ModelPathForApi<? super T, V> modelPath) {
        return new ApiFieldBuilder<>(
                avClass, modelPath,
                BatchFunction.one(modelPath::get),
                BatchBiFunction.one((model, value) -> {
                    modelPath.set(model, value);
                    return null;
                })
        );
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения и записи
     * для вложенных объектов
     */
    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            Class<V> avClass,
            ModelPathForApi<? super T, V> modelPath,
            BatchBiFunction<T, V> setterCoreValue) {
        return new ApiFieldBuilder<>(
                avClass, modelPath,
                BatchFunction.one(modelPath::get),
                setterCoreValue
        );
    }

    public static <T extends Model, V> ApiFieldBuilder<T, V, V> convertibleApiField(
            ParametrizedJavaTypeHolder<V> avClass,
            ModelPathForApi<? super T, V> modelPath,
            BatchBiFunction<T, V> setterCoreValue) {
        return new ApiFieldBuilder<>(
                avClass, modelPath,
                BatchFunction.one(modelPath::get),
                setterCoreValue
        );
    }

    /**
     * Метод создающий {@link ApiFieldBuilder} для поля, которое предназначени только для чтения и записи
     * с кастомным преобазованием
     */
    public static <T extends Model, AV, CV> ApiFieldBuilder<T, AV, CV> convertibleApiField(
            Class<AV> avClass,
            ModelProperty<? super T, CV> modelProperty,
            BatchFunction<T, AV> getterApiValue,
            BatchBiFunction<T, AV> setterCoreValue) {
        return new ApiFieldBuilder<>(avClass, modelProperty, getterApiValue, setterCoreValue);
    }

    public static <T extends Model, AV, CV> ApiFieldBuilder<T, AV, CV> convertibleApiField(
            ParametrizedJavaTypeHolder<AV> avClass,
            ModelProperty<? super T, CV> modelProperty,
            BatchFunction<T, AV> getterApiValue,
            BatchBiFunction<T, AV> setterCoreValue) {
        return new ApiFieldBuilder<>(avClass, modelProperty, getterApiValue, setterCoreValue);
    }

    @Override
    public String toString() {
        return "ApiField{" +
                "jsonName='" + jsonName + '\'' +
                ", modelPath=" + modelPath +
                '}';
    }
}
